package poly.op;

import poly.*;
import poly.factory.*;

/**
 * Adds host polynomial to input polynomial.  Returns the resulting sum polynomial.
 * @author Dung X. Nguyen
 * @since Copyright 2002 - DXN - All rights reserved
*/
public class Add implements IPolyOp {

    private IPolyFact _fact;

    public Add(IPolyFact f) {
        _fact = f;
    }

    /**
    * Uses a helper visitor to add a constant to the input APolynomial.
    * The helper visitor is implemented as an anonymous inner class.
    * @param pol a Constant polynomial
    * @param rhs an IPoly
    * @return an IPoly representing the sum of poly and rhs.
    */
    public Object constCase (final IConstPoly pol, Object... rhs) {
        // Use anonymous class to add a constant to the right-had-side:
        return ((IPoly)rhs[0]).execute(new IPolyOp() {

            // Add the two constants together:
            public Object constCase (IConstPoly p, Object... nu) {
                return _fact.makeConstPoly(p.getLeadCoef() + pol.getLeadCoef());
            }

            // Leave the leading term of p alone and (recursively) add the
            // constant to the lower poly of p.
            public Object nonConstCase(INCPoly p, Object... nu) {
                IPoly lowSum = (IPoly)p.getLowerPoly().execute (this);
                return _fact.makeNCPoly(p.getLeadCoef(), p.getOrder(), lowSum);
            }
        });
    }

    /**
    * @param pol a non-constant APolynomial
    * @param rhs a APolynomial
    * @return an IPoly representing the sum of pol and rhs.
    */
    public Object nonConstCase(INCPoly pol, Object... rhs) {
        // Save stuffs in local variables to avoid repeated type casting and
        // method calls:
        IPoly p = (IPoly)rhs[0];
        double pCoef = p.getLeadCoef();
        int pOrder = p.getOrder();
        double polyCoef = pol.getLeadCoef();
        int polyOrder = pol.getOrder();
        IPoly lowPol = pol.getLowerPoly();

        if (pOrder < polyOrder) {
            return _fact.makeNCPoly(polyCoef, polyOrder,
                                              (IPoly)lowPol.execute (this, p));
        }
        if (pOrder == polyOrder) {
            IPoly lowSum = (IPoly)lowPol.execute(this,
                                                  ((INCPoly)p).getLowerPoly());
            if (polyCoef < -pCoef || -pCoef < polyCoef) {
                return _fact.makeNCPoly(polyCoef + pCoef, polyOrder, lowSum);
            }
            return lowSum;
        }
        return _fact.makeNCPoly(pCoef, pOrder,
                       (IPoly)pol.execute(this, ((INCPoly)p).getLowerPoly()));
    }
}

