package poly.op;

import poly.*;

/**
 * Muliply a polynomial p by the leading term of a non-constant polynomial q.
 * The lower order polynomial of q is ignored.
 * For example: p(x) * (a x^n + lower) = p(x) * a x^n + lower terms. We keep
 * p(x) * a x^n and ignore the lower terms.
 * This is used in multiplying polynomials together using the distributive law.
 * @author DXN, SW
 */
public class MultMono implements IPolyOp {

    private IPolyFact _fact;
    private IPoly _zero;

    public MultMono(IPolyFact f) {
        _fact = f;
        _zero = _fact.makeConstPoly(0.0);
    }

    /**
     * For example, host = 5 and q(x) = 3 x^2 + lowerQ.
     * Then host * 3 x^2 = (5*3) x^2 = 15 x^2.
     */
    public Object constCase(IConstPoly pol, Object... q) {
      if(pol.getLeadCoef() == 0.0) {
        // pol = 0, doesn't matter what q is.
        return _zero;
      }
      else {
        return _fact.makeNCPoly(((IPoly)q[0]).getLeadCoef() * pol.getLeadCoef(),
                                ((IPoly)q[0]).getOrder(), _fact.makeConstPoly(0));
      }
    }

    /**
     * For example, host = 5 x^7 + lowerHost and q(x) = 3 x^2 + lowerQ.
     * Then host * 3 x^2 = (5*3) x^7 + lowerHost* 3 x^2.
     */
    public Object nonConstCase(INCPoly pol, Object... q) {
        return _fact.makeNCPoly(((IPoly)q[0]).getLeadCoef() * pol.getLeadCoef(),
                                ((IPoly)q[0]).getOrder() + pol.getOrder(),
                                (IPoly)pol.getLowerPoly().execute (this, q));
    }
}