package poly.op;

import poly.*;
import poly.factory.*;

/**
 * Adds host polynomial to input polynomial.  Returns the resulting sum polynomial.
 * @author Dung X. Nguyen
 * @since Copyright 2002 - DXN - All rights reserved
*/
public class AddOp implements IPolyOp {

    IPolyFact _fact;

    public AddOp(IPolyFact f) {
        _fact = f;
    }

    /**
    * Uses a helper visitor to add a constant to the input APolynomial.
    * The helper visitor can be implemented as an anonymous inner class.
    * @param poly a Constant polynomial
    * @param rhs an IPoly
    * @return an IPoly representing the sum of poly and rhs.
    */
    public Object constCase (IConstPoly poly, Object... rhs) {
        IPoly p = (IPoly)rhs[0];
        return p.execute(new AddConst(_fact), new Double(poly.getLeadCoef()));
    }

    /**
    * @param pol a non-constant APolynomial
    * @param rhs a APolynomial
    * @return an IPoly representing the sum of pol and rhs.
    */
    public Object nonConstCase(INCPoly pol, Object... rhs) {
        IPoly p = (IPoly)rhs[0];
        double pCoef = p.getLeadCoef();
        int pOrder = p.getOrder();
        double polyCoef = pol.getLeadCoef();
        int polyOrder = pol.getOrder();
        IPoly lowPol = pol.getLowerPoly();
        if (pOrder < polyOrder) {
            return _fact.makeNCPoly(polyCoef, polyOrder,
                                              (IPoly)lowPol.execute (this, p));
        }
        if (pOrder == polyOrder) {
            IPoly lowSum = (IPoly)lowPol.execute(this,
                                                  ((INCPoly)p).getLowerPoly());
            if (polyCoef < -pCoef || -pCoef < polyCoef) {
                return _fact.makeNCPoly(polyCoef + pCoef, polyOrder, lowSum);
            }
            return lowSum;
        }
        return _fact.makeNCPoly(pCoef, pOrder,
                       (IPoly)pol.execute(this, ((INCPoly)p).getLowerPoly()));
    }
}

