package maze.visitor;

import maze.*;
import java.util.*;

/**
 * Returns a string representation of all the rooms connected to the
 * host room.
 */
public class MazeToString implements IRoomAlgo<String, Object> {
  public static final MazeToString Singleton = new MazeToString();
  private MazeToString() {}
  
  /**
   * Nothing to do.  Return the string representation of an empty room
   */
  public String emptyCase(IEmptyRoom host, Object... nu) {
    return host.toString();
  }
  
  /**
   * Returns the toString of the host and recursively, the toStrings of all
   * non-duplicated rooms connected to this room.
   */
  public String neCase(INERoom host, Object... nu) {
    // Allocate a storage for all the rooms already seen.
    final ArrayList<INERoom> seen = new ArrayList<INERoom>();  
    seen.add(host);  // Just saw this room.
    // add this room to the result so far.
    String result = host.toString() + "--> {"
      +host.exitNorth().getName() +", "+ host.exitEast().getName() +", "
      +host.exitSouth().getName() +", "+ host.exitWest().getName()+"}"; 
    // Define a helper algo that uses the common room storage for rooms 
    // that have already been seen.
    IRoomAlgo<String, Object> helper = new IRoomAlgo<String, Object>() {
      /**
       * Empty case: nothing to add to the result
       */
      public String emptyCase(IEmptyRoom h, Object... nu) {
        return "";   
      }
      
      /**
       * If not seen, adds this room to the result and recurs
       */
      public String neCase(INERoom h, Object... nu) {
        if(seen.contains(h)) {
          return "";  // Stop looking further if already seen.
        }
        else {
          seen.add(h);  // Remember having seen this room.
          // Add this room to the result.
          String r = "\n"+h.toString()+ "--> {"
            +h.exitNorth().getName() +", "+ h.exitEast().getName() +", "
            +h.exitSouth().getName() +", "+ h.exitWest().getName()+"}"; 
          r += h.exitNorth().execute(this); // Recur to the 4 children
          r += h.exitEast().execute(this);
          r += h.exitSouth().execute(this);
          r += h.exitWest().execute(this);        
          return r;
        }
      }
    };
    
    // Recur on all the exits of the room
    result += host.exitNorth().execute(helper);
    result += host.exitEast().execute(helper);
    result += host.exitSouth().execute(helper);
    result += host.exitWest().execute(helper);
    return result;
  }
}