package maze;

import java.util.*;
import trinket.*;
/**
 * Concrete instance of an IRoomFactory
 */
public class RoomFactory implements IRoomFactory {
  public static final RoomFactory Singleton = new RoomFactory();
  private RoomFactory() {}
  
  /**
   * Singleton instance of an empty room
   */
  private IEmptyRoom emptyRoom = new IEmptyRoom() {
    public String getName() { return toString();}
    public <R,P> R execute(IRoomAlgo<R,P> algo, P... inps) {
      return algo.emptyCase(this, inps);
    }
    public String toString() { return "[]";}
  };
  
  /**
   * Method from IRoomFactory to make an empty room.
   * Just returns the singleton instance of the empty room.
   */
  public IEmptyRoom makeEmptyRoom(){
    return emptyRoom;
  }
  
  /**
   * Private internal definition of a non-empty room
   */
  private class NERoom implements INERoom {
    // Note that these private variables are accessible by the outer factory class.
    private String name;
    private IRoomData data;
    private ITrinket trinket;
    private IRoom north;
    private IRoom east;
    private IRoom south;
    private IRoom west;
    
    /**
     * Note that this private constructor is accessible by the outer factory class.
     */
    private NERoom(String aName, IRoomData aData, ITrinket aTrinket, IRoom aNorth, IRoom aEast, IRoom aSouth, IRoom aWest) {
      name = aName;
      data = aData;
      trinket = aTrinket;
      north = aNorth;
      east = aEast;
      south = aSouth;
      west = aWest;
    }
    
    public String getName() { return name; }
    
    public IRoomData getData() { return data;}
    public void setData(IRoomData data) { this.data = data;}
    
    public ITrinket getTrinket() {
      return trinket;
    }
    public void setTrinket(ITrinket t) {
      trinket = t;
    }

    public IRoom exitNorth() { return north;}
    public IRoom exitEast() { return east;}
    public IRoom exitSouth() { return south;}
    public IRoom exitWest() { return west;}
    
    public <R,P> R execute(IRoomAlgo<R,P> algo, P... inps) {
      return algo.neCase(this, inps);
    }
    
    /**
     * toString does not recursively print the child rooms because of potential
     * problems with loops in the data structure.
     */
    public String toString() {
      return "["+name+": "+data+": "+trinket+"]" ;
    }
  }
  
  /**
   * Method from IRoomFactory to make a non-empty room
   */
  public INERoom makeNERoom(String name, IRoomData data, ITrinket trinket, IRoom north, IRoom east, IRoom south, IRoom west){
    return new NERoom(name, data, trinket, north, east, south, west);
  }
  
  /**
   * Method from IRoomFactory to return the first (start) room for a set of randomly
   * connected rooms.
   * The returned room is non-empty and it's data is IRoomData.START.
   * @param maxRooms  The maximum number of interconnected rooms starting at the returned room.
   */
  public IRoom makeMaze(final int maxRooms) {
    final ArrayList<IRoom> roomSet = new ArrayList<IRoom>(); // internal storage for all the possible rooms
    
    // Make the maximum number of rooms, all disconnected from each other
    // for now.
    for(int i=0; i<maxRooms; i++) {
      if(1>(int)maxRooms*Math.random()) {
        roomSet.add(emptyRoom);
      }
      else {
        NERoom room = new NERoom("Room #"+i, IRoomData.UNSEEN, getRandomTrinket(),
                               emptyRoom,emptyRoom, emptyRoom, emptyRoom);
        roomSet.add(room);
      }
    }    
    final Random rand = new Random();  // Initialize the random number generator 
    
    // Walk through all the rooms that were made and set their neighbors to be 
    // random other existing rooms.
    for(IRoom r : roomSet) {
      r.execute(new IRoomAlgo<Object, Object>() {
          public Object emptyCase(IEmptyRoom host, Object... nu) {
            return null;
          }
          public Object neCase(INERoom host, Object... nu){
            NERoom neHost = (NERoom)host;
            neHost.north = roomSet.get(rand.nextInt(maxRooms));
            neHost.east = roomSet.get(rand.nextInt(maxRooms));
            neHost.south = roomSet.get(rand.nextInt(maxRooms));
            neHost.west = roomSet.get(rand.nextInt(maxRooms));
            return null;
          }
      });
    }
    setEndRoom(roomSet);
    return roomSet.get(0);
  }

  private void setEndRoom(final ArrayList<IRoom> roomSet) {
    final Random rand = new Random();  // Initialize the random number generator 
    
    // Randomly choose the end room.
    roomSet.get(rand.nextInt(roomSet.size())).execute(new IRoomAlgo<Object, Object> (){      
          public Object emptyCase(IEmptyRoom host, Object... nu) {
            setEndRoom(roomSet);
            return null;
          }
          public Object neCase(INERoom host, Object... nu){
            host.setData(IRoomData.END);
            return null;
          }
    });
  }
  
  private ITrinket[] trinkets = {ITrinket.NULL, ITrinket.COIN, ITrinket.WAND, ITrinket.FOOD, ITrinket.KEY};
  
  public ITrinket getRandomTrinket() {
    return trinkets[(int)(trinkets.length*Math.random())];
  }
}
