
package brs;

/**
 * Represents the state of the owner binary tree structure.  Union pattern
 * @author Dung X. Nguyen - Copyright 2001 - All rights reserved.
 */
abstract class ANode {
    /**
     * Gets the root data of the owner tree if it exists.
     * @param owner the BiTree that holds this node.
     * @return the data element of this node if it exists.
     * @exception NoSuchElementException if the owner is empty.
     */
    abstract Object getRootDat(BiTree owner);
    
    /**
     * Sets the root element of the owner tree to a given data object.
     * @param dat
     * @param owner the BiTree that holds this node.
     * @exception NoSuchElementException if the owner is empty.
     */
    abstract void setRootDat(Object dat, BiTree owner);
    
    /**
     * Gets the left subtree of the owner tree.
     * @param owner the BiTree that holds this node.
     * @return the left subtree of this node if it exists.
     * @exception NoSuchElementException if the owner is empty.
     */
    abstract BiTree getLeftSubTree(BiTree owner);
    
    /**
     * Gets the right subtree of the owner tree.
     * @param owner the BiTree that holds this node.
     * @return the right subtree of this node if it exists.
     * @exception NoSuchElementException if the owner is empty.
     */
    abstract BiTree getRightSubTree(BiTree owner);
    
    /**
     * Sets the left subtree of the owner tree to a given tree.
     * @param biTree != null.
     * @param owner the BiTree that holds this node.
     * @exception NoSuchElementException if the owner is empty.
     */
    abstract void setLeftSubTree(BiTree biTree, BiTree owner);
    
    /**
     * Sets the right subtree of the owner tree to a given tree.
     * @param biTree != null.
     * @param owner the BiTree that holds this node.
     * @exception NoSuchElementException if the owner is empty.
     */
    abstract void setRightSubTree(BiTree biTree, BiTree owner);
    
    /**
     * Inserts a root element to the owner tree. Allows the owner tree to change
     * state from empty to non-empty.
     * @param dat
     * @param owner the BiTree that holds this node.
     * @exception IllegaStateException if the owner is not empty.
     */
    abstract void insertRoot(Object dat, BiTree owner);
    
    /**
     * Removes and returns the root element from the owner tree.  Allows the
     * owner tree to change state from non-empty to empty.
     * @param dat
     * @param owner the BiTree that holds this node.
     * @exception IllegaStateException if the owner has more than one element.
     */
    abstract Object remRoot(BiTree owner);
    
    /**
     * Calls the appropriate visitor's method to execute the visiting algorithm.
     * @param algo the visiting algorithm
     * @param inp the input the algorithm needs.
     * @param owner the BiTree that holds this node.
     * @return the output for the algorithm.
     */
    abstract Object execute(BiTree owner, IVisitor algo, Object... inp);
}

