package tokens;

/**
 * Number token class.
 */
public class NumToken extends AToken {
    /**
     * Interface for a visitor to visit a number token.
     */
    public static interface INumVisitor extends ITokVisitor {
        /**
         * Case for a number token.
         *
         * @param host  number token
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object numCase(NumToken host, Object param);
    }

    /**
     * Abstract class for a chained visitor to visit a number token. In all non-plus cases, the successor is called.
     */
    public static abstract class AChainVis implements INumVisitor {
        /**
         * Successor.
         */
        private ITokVisitor _successor;

        /**
         * Constructor.
         *
         * @param successor visitor to serve as successor in the chain of responsibility.
         */
        protected AChainVis(ITokVisitor successor) {
            _successor = successor;
        }

        /**
         * Continue in the chain of responsibility.
         *
         * @param host  token
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object defaultCase(AToken host, Object param) {
            return host.execute(_successor, param);
        }
    }

    /**
     * Constructor for a number token.
     *
     * @param lexeme number lexeme
     */
    public NumToken(String lexeme) {
        super(lexeme);
    }

    /**
     * Execute the number case of the visitor.
     *
     * @param algo  visitor
     * @param param visitor-specific parameter
     * @return visitor-specific return value
     */
    public Object execute(ITokVisitor algo, Object param) {
        return (algo instanceof INumVisitor) ?
               ((INumVisitor) algo).numCase(this, param) :
               algo.defaultCase(this, param);
    }
}

