package treeN;
import java.util.*;

/**
 * Tree structure whose root node can contains an arbitrary number n data elements amd a
 * a corresponding n+1 children subtrees.
 * @author DXN
 * @author SBW
 */
public class TreeN {
    private Vector<TreeN> _children = new Vector<TreeN>();
    private Vector<Object> _data = new Vector<Object>();
    
    /**
     * Initializes this TreeN to an empty tree.
     */
    public TreeN() {
    }
    
    /**
     * Initializes this TreeN to a tree with one (Integer) element.
     */
    public TreeN(Integer n) { 
        this(new TreeN(), n, new TreeN());
    }
    
    /**
     * Private initialization of the _data and _children arrays.
     */
    private TreeN(Vector<Object> data, Vector<TreeN> children) {
        _data = data; _children = children;
    }
    
    /**
     * Private iniitalization to a tree with one element at the root and
     * a (left) subtree at index 0 and a (right) subtree at index 1.
     */
    private TreeN(TreeN lTree, Object n, TreeN rTree) {
        _data.add(n); _children.add(lTree); _children.add(rTree);
    }
    
    /**
     * Returns the data at a given index.  The left-most data is at index 0.
     * The right-most data is at index _data.size()-1.
     */
    public Object getDat(int idx) { 
        return _data.get(idx); 
    }
    
    /**
     * Returns the child subtree at a given index.  
     * The left-most child is at index 0.
     * The right-most child is at index _children.size()-1.
     */
    public TreeN getChild(int idx) { 
        return (TreeN) _children.get(idx); 
    }
    
    /**
     * Replace the data at idx with a new data object and return the old data.
     * This is added for convenience (and improved performance).
     */
    public Object setDat(int idx, Object e) {
        Object r = _data.get(idx);
        _data.set(idx, e);
        return r;
    }
    
    /**
     * Joins the supplied source tree t to the receiver at index i: the ith child 
     * of the receiver is deleted and the root node of t is spliced between the 
     * ith and i+1th elements of the receiver.  The children of t remain in their 
     * respective places with regards to the original elements of t.  
     * Splicing an empty source tree into a non-empty tree is a no-operation.  
     * Splicing a non-empty source tree into an empty tree will mutate the empty receiver 
     * tree into a shallow copy of the source tree.
     * See Figure 2 of DP4SBT paper.
     * @param idx -- 0 <= idx <= state of the tree+1, unless state = 0 then idx must equal 0.
     * @param tree -- tree to be spliced in.   If the state of this tree = 0, then nothing is done.
     * @return this.
     */
    public TreeN spliceAt(int idx, TreeN tree) {
        int i=tree._data.size();
        if(i>0) { // do nothing if tree is empty.
            if(_data.size() > 0) {
                _children.set(idx, tree.getChild(i--));  // replace the tree @ idx
            }
            else { // receiver is empty
                _children.add(idx,tree.getChild(i--));  // idx must = 0
            }
            for(; i>=0;i--) {
                _data.add(idx,tree.getDat(i));
                _children.add(idx, tree.getChild(i));
            }
        }
        return this;
    }
    
    /**
     * Mutates the receiver, in state s, into a 2-node tree (state = 1), 
     * where the ith element becomes the root data and the left childs root 
     * contains the 0 through i-1 elements of the original root and 
     * the right childs root contains the i+1 through s elements of the original root.  
     * Splitting up on an empty tree is a no-operation.
     * See Figure 2 of DP4SBT paper.
     * @param idx 0 <= idx <= state of tree - 1.
     * @return this.
     */
    public TreeN splitUpAt(int idx) {
        if(_data.size()>1) {
            TreeN lTree, rTree;
            Vector<Object> newData = new Vector<Object>();
            Vector<TreeN> newChildren = new Vector<TreeN>();
            Object rootDat = _data.remove(idx);
            for(int i = 0;i<idx;i++) {
                newData.add(_data.remove(0));
                newChildren.add(_children.remove(0));
            }
            newChildren.add(_children.remove(0));
            if(newData.size()>0) lTree = new TreeN(newData,newChildren);
            else lTree = (TreeN) newChildren.firstElement();
            if(_data.size()>0) rTree = new TreeN(_data, _children);
            else rTree = (TreeN) _children.firstElement();
            (_data = new Vector<Object>()).add(rootDat);
            (_children = new Vector<TreeN>()).add(lTree);
            _children.add(rTree);
        }
        return this;
    }
    
    /**
     * Removes the ith element from the root of the receiver including its corresponding 
     * left and right child trees.  The resultant new child tree is a 2-node tree where i
     * ts root data is the original ith element and where its left and right children are 
     * the original ith elements left and right children respectively.  
     * Splitting down a 2-node tree results in an empty tree and is equivalent to a 
     * deletion of a single data element.  
     * Splitting down an empty tree is a no-operation.
     * See Figure 2 of DP4BST paper.
     * @param idx 0 <= idx <= state of tree - 1.
     * @return this.
     */
    public TreeN splitDownAt(int idx) {
        if(_data.size()>1){
            TreeN newChild = new TreeN(getChild(idx),_data.remove(idx),getChild(idx+1));
            _children.remove(idx);
            _children.set(idx, newChild);
        }
        else {
            _data.clear();
            _children.clear();
        }
        return this;
    }
    
    /**
     * Hook method for the visitor pattern.
     */
    public Object execute(ITreeNAlgo algo, Object... param) {
        return algo.caseAt(_data.size(), this, param);
    }
}

