
import parser.*;
import parser.visitor.*;

import token.*;
import token.tokenizer.*;

import model.*;
import listFW.*;
import listFW.factory.*;
import rac.*;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.WindowEvent;
import java.io.FileNotFoundException;

public class GraphFrame extends JFrame {
    /**
     * Content pane.
     */
    private JPanel contentPane;
    
    /**
     * Output text area.
     */
    private JTextArea outputTA = new JTextArea();
    
    /**
     * Top panel.
     */
    private JPanel topPanel = new JPanel(new GridLayout(3,2));
    
    /**
     * Bottom scroll pane.
     */
    private JScrollPane scrollPane = new JScrollPane();
    
    /**
     * File name input field.
     */
    private JTextField inpFileTF = new JTextField();
    
    /**
     * Button to parse original grammar file.
     */
    private JButton parseGraphBtn = new JButton();
    
    /**
     * Button to test a grammar, not from a file.
     */
    private JButton testGraphBtn = new JButton();
    
    /**
     * Start list.
     */
    private JComboBox startList = new JComboBox();

    /**
     * End list.
     */
    private JComboBox endList = new JComboBox();

    /**
     * Button to search the graph using depth-first.
     */
    private JButton dfSearchGraphBtn = new JButton();

    /**
     * Button to search the graph using breadth-first.
     */
    private JButton bfSearchGraphBtn = new JButton();
    
    /**
     * Parsing result.
     */
    IGrammarSymbol result;
    
    /**
     * Tokenizer.
     */
    private ITokenizer tok;
    
    /**
     * Graph.
     */
    private Graph _g = new Graph();
    
    /**
     * Constructor for the frame.
     */
    public GraphFrame() {
        enableEvents(AWTEvent.WINDOW_EVENT_MASK);
        try {
            jbInit();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    /**
     * Initialize GUI components.
     */
    private void jbInit() {
        contentPane = (JPanel) this.getContentPane();
        contentPane.setLayout(new BorderLayout());
        
        this.setSize(new Dimension(616, 317));
        this.setTitle("Frame Title");
        
        inpFileTF.setMinimumSize(new Dimension(100, 21));
        inpFileTF.setPreferredSize(new Dimension(100, 21));
        inpFileTF.setText("graph1.txt");
        
        parseGraphBtn.setText("Parse Graph");
        parseGraphBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(ActionEvent e) {
                parseGraphBtn_actionPerformed(e);
            }
        });
        testGraphBtn.setText("Test");
        testGraphBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(ActionEvent e) {
                testGraphBtn_actionPerformed(e);
            }
        });
        
        startList.removeAllItems();
        endList.removeAllItems();
        
        dfSearchGraphBtn.setText("Depth First");
        dfSearchGraphBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(ActionEvent e) {
                searchGraphBtn_actionPerformed(e, new LRSStackFactory<Node>());
            }
        });
        bfSearchGraphBtn.setText("Breadth First");
        bfSearchGraphBtn.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(ActionEvent e) {
                searchGraphBtn_actionPerformed(e, new LRSQueueFactory<Node>());
            }
        });
        
        JPanel graphButtonPanel = new JPanel(new GridLayout(1,2));
        topPanel.add(inpFileTF);
        graphButtonPanel.add(parseGraphBtn);
        graphButtonPanel.add(testGraphBtn);
        topPanel.add(graphButtonPanel);
        topPanel.add(startList);
        topPanel.add(endList);
        topPanel.add(dfSearchGraphBtn);
        topPanel.add(bfSearchGraphBtn);
        
        contentPane.add(scrollPane, BorderLayout.CENTER);
        contentPane.add(topPanel, BorderLayout.NORTH);
        scrollPane.getViewport().add(outputTA, null);
    }
    
    /**
     * Overridden so we can exit when window is closed.
     *
     * @param e window event
     */
    protected void processWindowEvent(WindowEvent e) {
        super.processWindowEvent(e);
        if (e.getID() == WindowEvent.WINDOW_CLOSING) {
            System.exit(0);
        }
    }
    
    /**
     * Make original grammar parser.
     *  
     *  G ::= H | Empty
     *  H ::= N ; G
     *  N ::= IdToken -> E
     *  E ::= F | Empty
     *  F ::= IdToken E
     * 
     * @throws FileNotFoundException thrown if file is not found
     */
    protected ITokVisitorFact makeOrigParser() throws FileNotFoundException {
        tok = new Tokenizer1(inpFileTF.getText());
        
        // --------------------------------------------------------------------
        // TODO: 1
        // Write the code to parse the grammar above using
        // SequenceFact, MultiSequenceFact, CombinationFact, TerminalSymbolFact,
        // MTSymbolFact, and/or ProxyFact.
        
        ProxyFact eFact_Proxy = new ProxyFact();
        ITokVisitorFact eFact = new CombinationFact("E",
                                                    tok,
                                                    new SequenceFact("F",
                                                                     tok,
                                                                     new TerminalSymbolFact("Id", tok),
                                                                     eFact_Proxy),
                                                    new MTSymbolFact(tok));
        eFact_Proxy.setFact(eFact);
        
        ProxyFact gFact_Proxy = new ProxyFact();
        ITokVisitorFact gFact = new CombinationFact("G",
                                                    tok, 
                                                    new MultiSequenceFact("H",
                                                                          tok, 
                                                                          new MultiSequenceFact("N",
                                                                                                tok,
                                                                                                new TerminalSymbolFact("Id", tok),
                                                                                                new TerminalSymbolFact("->", tok),
                                                                                                eFact),
                                                                          new TerminalSymbolFact(";", tok),
                                                                          gFact_Proxy),
                                                    new MTSymbolFact(tok));
        gFact_Proxy.setFact(gFact);
        
        System.err.println("Parser Factory = " + gFact);
        
        return gFact;
        // --------------------------------------------------------------------
    }
    
    /**
     * ToString visitor for grammar.
     */
    static final AGramSymVisitor<String,String> toStringHelp = new AGramSymVisitor<String, String>(new IGramSymVisitorCmd<String,String>() {
        public String apply(String index, IGrammarSymbol host, String... params) {
            // terminal case
            TerminalSymbol h = (TerminalSymbol)host;
            return "|_ "+h.getClass().getName()+": "+h.toString();
        }
    }) { };
    static {
        toStringHelp.setCmd("Sequence", new IGramSymVisitorCmd<String,String>() {
            public String apply(String index, IGrammarSymbol host, String... params) {
                // sequence case
                SequenceSymbol h = (SequenceSymbol)host;
                return "|_ "+h.getClass().getName()+"\n"+
                    params[0]+h.getSymbol1().execute(toStringHelp,params[0]+"|  ")+"\n"+
                    params[0]+h.getSymbol2().execute(toStringHelp,params[0]+"  ");
            }
        });
        toStringHelp.setCmd("MTSymbol", new IGramSymVisitorCmd<String,String>() {
            public String apply(String index, IGrammarSymbol host, String... params) {
                // empty case
                return "|_ "+host.getClass().getName();
            }
        });
    }
    static final AGramSymVisitor<String,String> toString = new AGramSymVisitor<String, String>(new IGramSymVisitorCmd<String,String>() {
        public String apply(String index, IGrammarSymbol host, String... params) {
            // terminal case
            TerminalSymbol h = (TerminalSymbol)host;
            return h.getClass().getName()+": "+h.toString();
        }
    }) {
        {
            setCmd("MTSymbol", new IGramSymVisitorCmd<String,String>() {
                public String apply(String index, IGrammarSymbol host, String... params) {
                    // empty case
                    return host.getClass().getName();
                }
            });
            setCmd("Sequence", new IGramSymVisitorCmd<String,String>() {
                public String apply(String index, IGrammarSymbol host, String... params) {
                    // sequence case
                    SequenceSymbol h = (SequenceSymbol)host;
                    return params[0]+h.getClass().getName()+"\n"+
                        h.getSymbol1().execute(toStringHelp,"|  ")+"\n"+
                        h.getSymbol2().execute(toStringHelp,"   ");
                }
            });
        }
    };
    
    /**
     * Create a hard-coded graph, not parsed from a file.
     * You can use this to test part 3 of the exam in case you don't
     * get parts 1 or 2 working.
     *
     * @param e action event
     */
    @SuppressWarnings("unchecked")
    void testGraphBtn_actionPerformed(ActionEvent e) {
        startList.removeAllItems();
        endList.removeAllItems();
        
        StringBuilder sb = new StringBuilder();
        sb.append("This graph was not parsed from a file, but was hard-coded:\n\n");
        
        Node outside = new Node("Outside");
        Node kitchen = new Node("Kitchen");
        Node bathroom = new Node("Bathroom");
        Node bedroom = new Node("Bedroom");
        Node closet = new Node("Closet", bedroom);
        Node livingRoom = new Node("LivingRoom", outside, kitchen, bathroom, bedroom);
        bedroom.getNeighbors().add(closet);
        outside.getNeighbors().add(livingRoom);
        kitchen.getNeighbors().add(livingRoom);
        bathroom.getNeighbors().add(livingRoom);
        bedroom.getNeighbors().add(livingRoom);
        _g = new Graph(outside, livingRoom, kitchen, bathroom, bedroom, closet);
        sb.append(_g);
        
        outputTA.setText(sb.toString());
        
        for(Node n: _g.getNodes().values()) {
            startList.addItem(n);
            endList.addItem(n);
        }
        if (_g.getNodes().size()>0) {
            startList.setSelectedIndex(0);
            endList.setSelectedIndex(0);
        }
    }
        
    /**
     * Parse file.
     *
     * @param e action event
     */
    @SuppressWarnings("unchecked")
    void parseGraphBtn_actionPerformed(ActionEvent e) {
        try {
            startList.removeAllItems();
            endList.removeAllItems();
            ITokVisitorFact origParseFact = makeOrigParser();
            
            ITokVisitor<IGrammarSymbol, Object> parser = origParseFact.makeVisitor();
            System.err.println("Parser visitor = " + parser);
            
            result = tok.getNextToken().execute(parser);
            System.err.println("Result = " + result);
            
            StringBuilder sb = new StringBuilder();
            sb.append(result.toString());
            sb.append('\n');
            sb.append(result.execute(toString, ""));
            sb.append("\n\n");
            
            // --------------------------------------------------------------------
            // TODO: 2b
            // Convert result (result variable) from parser into a Graph (_g field)
            // using ToModelAlgo visitor.
            _g = result.execute(ToModelAlgo.Singleton);
            sb.append(_g);
            // --------------------------------------------------------------------
            
            outputTA.setText(sb.toString());
            
            for(Node n: _g.getNodes().values()) {
                startList.addItem(n);
                endList.addItem(n);
            }
            if (_g.getNodes().size()>0) {
                startList.setSelectedIndex(0);
                endList.setSelectedIndex(0);
            }
        }
        catch (Exception e1) {
            StringBuffer sb = new StringBuffer();
            sb.append(e1.toString());
            sb.append('\n');
            StackTraceElement[] ste = e1.getStackTrace();
            for (int i = 0; i < ste.length; ++i) {
                sb.append('\t');
                sb.append(ste[i].toString());
                sb.append('\n');
            }
            outputTA.setText(sb.toString());
            outputTA.setCaretPosition(0);
            e1.printStackTrace();
        }
    }
    
    /**
     * Search graph.
     *
     * @param e action event
     */
    void searchGraphBtn_actionPerformed(ActionEvent e, IRACFactory<Node> fac) {
        if (_g.getNodes().size()>0) {
            StringBuilder sb = new StringBuilder(outputTA.getText());
            sb.append("\n\n");
            Node start = _g.getNodes().get(startList.getSelectedItem().toString());
            Node end = _g.getNodes().get(endList.getSelectedItem().toString());
            IList<Node> path = new GraphSearch(fac,new CompositeListFactory<Node>()).search(_g, start, end);
            sb.append(path);
            outputTA.setText(sb.toString());
        }
    }
}