package model;

import extvisitor.*;
import rac.*;
import listFW.*;
import listFW.visitor.*;

import java.util.*;

/** Extended visitor that performs the graph processing. */
public abstract class GraphVisitor<R,P,S> extends AExtVisitor<R,String,P,Node> {
    /**
     * Constructor that takes a default command to use.
     * @param defaultCmd  The default command to use.
     */
    public GraphVisitor(NodeCommand<R,P,S> defaultCmd) {
        super(defaultCmd);
    }
    
    /**
     * Constructor that takes a value that the default command will return.  
     * A default command is created will return the given value and do nothing else.
     * @param noOpResult The value for the default command to return.
     */
    public GraphVisitor(final R noOpResult) {
        super(new NodeCommand<R,P,S>(null) {
            public R apply(String index, Node host, P... params) {
                return noOpResult;
            }
        });
    }
    
    /**
     * Copy constructor for the class.   Copies all cmds with their index keys 
     * including the default cmd.
     * @param other The visitor from which the cmds will be copied.
     */
    public GraphVisitor(GraphVisitor<R,P,S> other) {
        super(other);
    }
    
    /**
     * Associates the given index value with the given command
     * @param idx The index value to use associate with the command.
     * @param cmd The command associated with the index value
     */
    public void setCmd(String idx, NodeCommand<R,P,S> cmd) {
        super.setCmd(idx, cmd);
    }

    /**
     * Associates the given index value with the given command
     * @param idx The index value to use associate with the command.
     * @param cmd The command associated with the index value
     */
    public void setCmd(String idx, IExtVisitorCmd<R, String, P, Node> cmd) {
        throw new IllegalArgumentException("Command must be a NodeCommand.");
    }
    
    /**
     * Retrieve the current default command
     * @return The current default command
     */
    @SuppressWarnings("unchecked")
    public NodeCommand<R,P,S> getDefaultCmd(){
        return (NodeCommand<R,P,S>)super.getDefaultCmd();
    }
    
    /**
     * Set the default command
     * @param defaultCmd The new default command
     */
    public void setDefaultCmd(NodeCommand<R,P,S> defaultCmd) {
        super.setDefaultCmd(defaultCmd);
    }

    /**
     * Set the default command
     * @param defaultCmd The new default command
     */
    public void setDefaultCmd(IExtVisitorCmd<R, String, P, Node> defaultCmd) {
        throw new IllegalArgumentException("Command must be a NodeCommand.");
    }

    /** Overridden getCmd method to return NodeCommands.
      * @param idx an index value
      * @return NodeCommand associated with the given index value. */
    @SuppressWarnings("unchecked")
    public NodeCommand<R,P,S> getCmd(String idx) { return (NodeCommand<R,P,S>)super.getCmd(idx); }

    /** Abstract base class for node commands.
      * Can store and update the state of the node. */
    public static abstract class NodeCommand<R,P,S> implements IExtVisitorCmd<R,String,P,Node> {
        /** State of this node. */
        private S _state;
        /** Create a new NodeCommand with the specified state.
          * @param s state of node */
        public NodeCommand(S s) { _state = s; }
        /** Get the state of this node.
          * @return state of this node */
        public S getState() { return _state; }
        /** Set the state of this node.
          * @param s state of this node */
        public void setState(S s) { _state = s; }
        /** Process this node. */
        public abstract R apply(String idx, Node host, P... params);
    }
    
    /** Process the next node. */
    protected abstract R processNext();
}
