package model;

import extvisitor.*;
import java.util.*;

/**
 * This class represents a node in a graph.
 * A node has a name and a set of neighbor nodes to which it is connected.
 * Note that the edges that connect two nodes are directed, i.e. that
 * if there is an edge from A to B, there does not need to be an edge from
 * B to A. A node can also be its own neighbor, i.e. there is an edge
 * that loops back to the node itself.
 * The name of a node must be unique in a graph.
 */
public class Node implements IExtVisitorHost<String,Node> {
    /** The name of this node. */
    protected String _name;
    /** The set of neighbors. */
    protected HashSet<Node> _neighbors = new HashSet<Node>();
    
    /** Create a new node with the given name and no neighbors.
      * @param name name of the node */
    public Node(String name) { _name = name; }
    /** Create a new node with the given name and neighbors.
      * @param name name of the node
      * @param neighbors neighbors of the node */
    public Node(String name, Node... neighbors) {
        this(name);
        for(Node n: neighbors) { _neighbors.add(n); }
    }

    /** Return the name.
      * @return name of the node */
    public String getName() { return _name; }
    
    /** Return the set of neighbors.
      * @return set of neighbors */
    public HashSet<Node> getNeighbors() {
        return _neighbors;
    }
    
    /** Return true if this node is equal to the other, where
      * "equals" means has the same name.
      * @param other other node
      * @return true if the same name */
    public boolean equals(Object other) {
        if (!(other instanceof Node)) return false;
        Node o = (Node)other;
        return o._name.equals(_name);
    }
    
    /** Return a hash code for this node, based on the name.
      * @return hash code */
    public int hashCode() { return _name.hashCode(); }

    /** Execute the visitor with the specified arguments.
      * @param algo visitor
      * @param inps visitor-specific input
      * @return visitor-specific output */
    public <R, P> R execute(IExtVisitor<R, String, P, Node> algo, P... inps){
        return algo.caseAt(_name, this, inps);
    }

    /** String representation of this node.
      * @return name of this node */
    public String toString() {
        return _name;
    }
}
