package parser;

import token.*;
import token.tokenizer.*;
import extvisitor.*;
import fp.*;
import java.util.*;

/**
 * A factory to create visitors to parse a binary sequence of grammar tokens.
 * To easily create a sequence of two or more grammar tokens, use 
 * MultiSequenceFact rather than multiple SequenceFacts.
 */
public class SequenceFact extends ATokVisitorFact {
  
   /**
   * The name of this instance of the factory
   */
  private String _name;
  
  /**
   * Factory for the first grammar non-terminals.
   */
  private ITokVisitorFact _fact1;
  
  /**
   * Visitor for the first grammar non-terminals.
   */
  private ATokVisitor<IGrammarSymbol, Object> _parse1;
  
  /**
   * Factory for the second grammar non-terminals.
   */
  private ITokVisitorFact _fact2;
  
  /**
   * Visitor for the second grammar non-terminals.
   */
  private ATokVisitor<IGrammarSymbol, Object> _parse2;
  
  /**
   * Initializer lambda for this factory.
   */
  private ILambda<Object, Object> _initializer = new ILambda<Object, Object>() {
    public Object apply(Object... nu) {
      // change state to no-op
      _initializer = new NoOpLambda<Object>();
      
      // initialize
      _parse1 = _fact1.makeVisitor();
      _parse2 = _fact2.makeVisitor();
      return null;
    }
  };
  
  /**
   * Constructor for the sequence factory,
   *
   * @param name The grammar token's name of this instance of a sequence.
   * @param tkz    tokenizer to use
   * @param fact1  factory for the first non-terminals
   * @param fact2 factory for the second non-terminals
   */
  public SequenceFact(String name, ITokenizer tkz, ITokVisitorFact fact1, ITokVisitorFact fact2) {
    super(tkz);
    _name = name;
    _fact1 = fact1;
    _fact2 = fact2;
  }
  
  
  /**
   * Make a token visitor to parse an sequence non-terminal.
   *
   * @return token visitor
   */
  public ATokVisitor<IGrammarSymbol, Object> makeVisitor() {
    initialize();
    
    return new ATokVisitor<IGrammarSymbol, Object>(_parse1)
    {
      {
        // Decorate every command that was copied from _parse1
        for(Map.Entry<String, IExtVisitorCmd<IGrammarSymbol, String, Object, Token>> entry : getCmds()) {
          final IExtVisitorCmd<IGrammarSymbol, String, Object, Token> origCmd = entry.getValue();
          entry.setValue(new ITokVisitorCmd<IGrammarSymbol, Object>() {
              public IGrammarSymbol apply(String idx, Token host, Object... inps) {
                System.out.println("Sequence: "+_name);
                return new SequenceSymbol(_name, origCmd.apply(idx, host, inps),  nextToken().execute(_parse2, inps));
              }
            });
        }
        
        final IExtVisitorCmd<IGrammarSymbol, String, Object, Token> oldDefaultCmd = getDefaultCmd();
        setDefaultCmd(new IExtVisitorCmd<IGrammarSymbol, String, Object, Token>() {
          public IGrammarSymbol apply(String idx, Token host, Object... inps) {
            return new SequenceSymbol(_name,(IGrammarSymbol)oldDefaultCmd.apply(idx, host, inps), (IGrammarSymbol) nextToken().execute(_parse2, inps));
          }
        });
      }
    };
  }
  
  /**
   * Make a token visitor that will process the combination of this sequence
   * or the other given symbol 
   *
   * @param other The visitor for the other symbol in the combination 
   * @return A token visitor
   */
  public ATokVisitor<IGrammarSymbol, Object> makeCombinedVisitor(ATokVisitor<IGrammarSymbol, Object> other) {
    initialize();
    
    final ATokVisitor<IGrammarSymbol, Object> v = makeVisitor();
    // Copy all cmds from other to v
    for(Map.Entry<String, IExtVisitorCmd<IGrammarSymbol, String, Object, Token>> otherEntry : other.getCmds()) {
      v.setCmd(otherEntry.getKey(), otherEntry.getValue());
    }
//    other.map(new AExtVisitor.IMapLambda<String, IExtVisitorCmd<IGrammarSymbol, String, Object, Token>>() {
//      public IExtVisitorCmd<IGrammarSymbol, String, Object, Token> apply(String idx, final IExtVisitorCmd<IGrammarSymbol, String, Object, Token> cmd) {
//        v.setCmd(idx, cmd);   // copy command to v
//        return cmd;  // put cmd back in other
//      }
//    });
    v.setDefaultCmd(other.getDefaultCmd());
    return v;
  }
  
  /**
   * Make the visitor.
   */
  private void initialize() {
    _initializer.apply();
  }
}

