package parser.visitor;

import model.*;
import parser.*;
import extvisitor.*;

/**
 * Turn the parse tree into an AST that is more easy to use.
 * 
 * Example:
 * LivingRoom->Kitchen Bedroom Bathroom Outside;
 * Outside->LivingRoom;
 * Kitchen->LivingRoom;
 * Bedroom->LivingRoom Closet;
 * Closet->Bedroom;
 * Bathroom->LivingRoom;
 * 
parser.SequenceSymbol
|_ parser.SequenceSymbol
|  |_ parser.TerminalSymbol: LivingRoom
|  |_ parser.SequenceSymbol
|    |_ parser.TerminalSymbol: ->
|    |_ parser.SequenceSymbol
|      |_ parser.TerminalSymbol: Kitchen
|      |_ parser.SequenceSymbol
|        |_ parser.TerminalSymbol: Bedroom
|        |_ parser.SequenceSymbol
|          |_ parser.TerminalSymbol: Bathroom
|          |_ parser.SequenceSymbol
|            |_ parser.TerminalSymbol: Outside
|            |_ parser.MTSymbol
|_ parser.SequenceSymbol
   |_ parser.TerminalSymbol: ;
   |_ parser.SequenceSymbol
     |_ parser.SequenceSymbol
     |  |_ parser.TerminalSymbol: Outside
     |  |_ parser.SequenceSymbol
     |    |_ parser.TerminalSymbol: ->
     |    |_ parser.SequenceSymbol
     |      |_ parser.TerminalSymbol: LivingRoom
     |      |_ parser.MTSymbol
     |_ parser.SequenceSymbol
       |_ parser.TerminalSymbol: ;
       |_ parser.SequenceSymbol
         |_ parser.SequenceSymbol
         |  |_ parser.TerminalSymbol: Kitchen
         |  |_ parser.SequenceSymbol
         |    |_ parser.TerminalSymbol: ->
         |    |_ parser.SequenceSymbol
         |      |_ parser.TerminalSymbol: LivingRoom
         |      |_ parser.MTSymbol
         |_ parser.SequenceSymbol
           |_ parser.TerminalSymbol: ;
           |_ parser.SequenceSymbol
             |_ parser.SequenceSymbol
             |  |_ parser.TerminalSymbol: Bedroom
             |  |_ parser.SequenceSymbol
             |    |_ parser.TerminalSymbol: ->
             |    |_ parser.SequenceSymbol
             |      |_ parser.TerminalSymbol: LivingRoom
             |      |_ parser.SequenceSymbol
             |        |_ parser.TerminalSymbol: Closet
             |        |_ parser.MTSymbol
             |_ parser.SequenceSymbol
               |_ parser.TerminalSymbol: ;
               |_ parser.SequenceSymbol
                 |_ parser.SequenceSymbol
                 |  |_ parser.TerminalSymbol: Closet
                 |  |_ parser.SequenceSymbol
                 |    |_ parser.TerminalSymbol: ->
                 |    |_ parser.SequenceSymbol
                 |      |_ parser.TerminalSymbol: Bedroom
                 |      |_ parser.MTSymbol
                 |_ parser.SequenceSymbol
                   |_ parser.TerminalSymbol: ;
                   |_ parser.SequenceSymbol
                     |_ parser.SequenceSymbol
                     |  |_ parser.TerminalSymbol: Bathroom
                     |  |_ parser.SequenceSymbol
                     |    |_ parser.TerminalSymbol: ->
                     |    |_ parser.SequenceSymbol
                     |      |_ parser.TerminalSymbol: LivingRoom
                     |      |_ parser.MTSymbol
                     |_ parser.SequenceSymbol
                       |_ parser.TerminalSymbol: ;
                       |_ parser.MTSymbol
 */
public class ToModelAlgo extends AGramSymVisitor</*R=*/Graph, /*P=*/Void> {
    public static final ToModelAlgo Singleton = new ToModelAlgo();
    
    private ToModelAlgo() {
        super(new IGramSymVisitorCmd<Graph,Void>() {
            public Graph apply(String idx, IGrammarSymbol host, Void... inps) {
                throw new IllegalArgumentException("Parse tree does not represent a correct graph.");
            }
        });
        setCmd("Sequence",new IGramSymVisitorCmd<Graph,Void>() {
            public Graph apply(String idx, IGrammarSymbol host, Void... inps) {
                return host.execute(Helper.Singleton, new Graph());
            }
        });
        setCmd("MTSymbol",new IGramSymVisitorCmd<Graph,Void>() {
            public Graph apply(String idx, IGrammarSymbol host, Void... inps) {
                return new Graph();
            }
        });
    }
    
    /**
     * This visitor converts a SequenceSymbol consisting of two other SequenceSymbols,
     * once representing the node branch, one consisting of the semicolon/rest branch,
     * to a Graph.
     * inps[0] is the Graph we are working on.
     * 
     * parser.SequenceSymbol
     * |_ parser.SequenceSymbol (Node branch)
     * |_ parser.SequenceSymbol (Semicolomn/rest branch)
     * or
     * parser.MTSymbol
     */
    private static class Helper extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Graph> {
        public static final Helper Singleton = new Helper();
        
        private Helper() {
            // --------------------------------------------------------------------
            // TODO: 2a
            setCmd("Sequence",new IGramSymVisitorCmd<Graph,Graph>() {
                public Graph apply(String idx, IGrammarSymbol host, Graph... inps) {
                    SequenceSymbol sh = (SequenceSymbol)host;
                    return sh.getSymbol2().execute(SemicolonRestToModel.Singleton, 
                                                   sh.getSymbol1().execute(NodeToModel.Singleton, inps));
                }
            });
            setCmd("MTSymbol",new IGramSymVisitorCmd<Graph,Graph>() {
                public Graph apply(String idx, IGrammarSymbol host, Graph... inps) {
                    return inps[0];
                }
            });
            // --------------------------------------------------------------------
        }  
    }
    
    /**
     * This visitor converts a SequenceSymbol consisting of a TerminalSymbol (semicolon)
     * and another SequenceSymbol representing the rest of the tree after the semicolon
     * to a Graph.
     * inps[0] is the Graph we are working on.
     * 
     * |_ parser.SequenceSymbol
     *   |_ parser.TerminalSymbol: ;
     *   |_ parser.SequenceSymbol (Rest branch)
     */
    private static class SemicolonRestToModel extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Graph> {
        public static final SemicolonRestToModel Singleton = new SemicolonRestToModel();
        
        private SemicolonRestToModel() {
            // --------------------------------------------------------------------
            // TODO: 2a
            setCmd("Sequence",new IGramSymVisitorCmd<Graph,Graph>() {
                public Graph apply(String idx, IGrammarSymbol host, Graph... inps) {
                    SequenceSymbol sh = (SequenceSymbol)host;
                    return sh.getSymbol2().execute(Helper.Singleton, 
                                                   sh.getSymbol1().execute(CheckSemicolon.Singleton, inps));
                }
            });
            // --------------------------------------------------------------------
        }  
    }
    
    /**
     * This visitor makes sure that the symbol is a semicolon.
     * inps[0] is the Graph we are working on.
     * 
     * |_ parser.TerminalSymbol: ;
     */
    private static class CheckSemicolon extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Graph> {
        public static final CheckSemicolon Singleton = new CheckSemicolon();
        
        private CheckSemicolon() {
            // --------------------------------------------------------------------
            // TODO: 2a
            setCmd(";", NOP);
            // --------------------------------------------------------------------
        }  
    }

    /**
     * This visitor converts a SequenceSymbol consisting of a TerminalSymbol (Id)
     * and another SequenceSymbol representing the rest of the tree after the Id
     * to a Graph.
     * inps[0] is the Graph we are working on.
     * Note that when we process the rest after the Id, we have to pass the name
     * of this node along as well, so when we use ArrowListToModel, we pass the
     * current graph as inps[0] and the name of the node as inps[1].
     * 
     * |_ parser.SequenceSymbol
     * |  |_ parser.TerminalSymbol: LivingRoom
     * |  |_ parser.SequenceSymbol (Arrow/list branch)
     */
    private static class NodeToModel extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Graph> {
        public static final NodeToModel Singleton = new NodeToModel();
        
        private NodeToModel() {
            // --------------------------------------------------------------------
            // TODO: 2a
            setCmd("Sequence",new IGramSymVisitorCmd<Graph,Graph>() {
                public Graph apply(String idx, IGrammarSymbol host, Graph... inps) {
                    SequenceSymbol sh = (SequenceSymbol)host;
                    Node n = sh.getSymbol1().execute(ProcessName.Singleton, inps[0]);
                    return sh.getSymbol2().execute(ArrowListToModel.Singleton, inps[0], n.getName());
                }
            });
            // --------------------------------------------------------------------
        }  
    }

    /**
     * This visitor processes a TerminalSymbol representing the Id to the left of an arrow
     * and returns a Node with that name.
     * inps[0] is the Graph we are working on.
     * Checks if the graph already contains a node with the name. If it does, this visitor
     * returns that node. If it doesn't, it adds a new Node to the graph and returns the
     * new node.
     * 
     * |_ parser.TerminalSymbol: LivingRoom
     */
    private static class ProcessName extends AGramSymThrowsVisitor</*R=*/Node, /*P=*/Graph> {
        public static final ProcessName Singleton = new ProcessName();
        
        private ProcessName() {
            // TODO: 2a
            // --------------------------------------------------------------------
            setCmd("Id", new IGramSymVisitorCmd<Node,Graph>() {
                public Node apply(String idx, IGrammarSymbol host, Graph... inps) {
                    Graph g = inps[0];
                    String nodeName = host.toString();
                    Node n = g.getNodes().get(nodeName);
                    if (n==null) {
                        // add node if it didn't already exist
                        g.getNodes().put(nodeName, n = new Node(nodeName));
                    }
                    return n;
                }
            });
            // --------------------------------------------------------------------
        }
    }

    /**
     * This visitor converts a SequenceSymbol consisting of a TerminalSymbol (arrow)
     * and another SequenceSymbol representing the list of neighbors after the arrow
     * to a Graph.
     * inps[0] is the Graph we are working on.
     * inps[1] is the name of the node whose neighbors are in the list
     * 
     * |_ parser.SequenceSymbol
     *   |_ parser.TerminalSymbol: ->
     *   |_ parser.SequenceSymbol (List branch)
     */
    private static class ArrowListToModel extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Object> {
        public static final ArrowListToModel Singleton = new ArrowListToModel();
        
        private ArrowListToModel() {
            // TODO: 2a
            setCmd("Sequence",new IGramSymVisitorCmd<Graph,Object>() {
                public Graph apply(String idx, IGrammarSymbol host, Object... inps) {
                    SequenceSymbol sh = (SequenceSymbol)host;
                    return sh.getSymbol2().execute(ListToModel.Singleton, 
                                                   sh.getSymbol1().execute(CheckArrow.Singleton, inps[0]), inps[1]);
                }
            });
        }  
    }
    
    /**
     * This visitor makes sure that the symbol is an arrow ->.
     * inps[0] is the Graph we are working on.
     * 
     * |_ parser.TerminalSymbol: ->
     */
    private static class CheckArrow extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Object> {
        public static final CheckArrow Singleton = new CheckArrow();
        
        private CheckArrow() {
            // --------------------------------------------------------------------
            // TODO: 2a
            setCmd("->", NOP_POBJECT);
            // --------------------------------------------------------------------
        }  
    }


    /**
     * This visitor converts a SequenceSymbol consisting of a terminal symbol (Id)
     * representing a neighbor, and another SequenceSymbols, representing the rest of the
     * neighbor list, to a Graph.
     * inps[0] is the Graph we are working on.
     * inps[1] is the name of the node whose neighbors are in the list
     * 
     * |_ parser.SequenceSymbol
     *   |_ parser.TerminalSymbol: Bathroom
     *   |_ parser.SequenceSymbol
     * or
     * parser.MTSymbol
     */
    private static class ListToModel extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Object> {
        public static final ListToModel Singleton = new ListToModel();
        
        private ListToModel() {
            // --------------------------------------------------------------------
            // TODO: 2a
            setCmd("MTSymbol", NOP_POBJECT);
            setCmd("Sequence",new IGramSymVisitorCmd<Graph,Object>() {
                public Graph apply(String idx, IGrammarSymbol host, Object... inps) {
                    SequenceSymbol sh = (SequenceSymbol)host;
                    return sh.getSymbol2().execute(ListToModel.Singleton, 
                                                   sh.getSymbol1().execute(ProcessNeighbor.Singleton, inps[0], inps[1]), inps[1]);
                }
            });
            // --------------------------------------------------------------------
        }  
    }
    
    /**
     * This visitor processes a TerminalSymbol representing the Id of a neighbor (to the
     * right of an arrow) and returns the graph, with that neighbor added to the node
     * with the name that was on the left of the arrow (inps[1]).
     * inps[0] is the Graph we are working on.
     * inps[1] is the name of the node whose neighbors are in the list
     * Checks if the graph already contains a node with the name. If it doesn't, this
     * visitor adds it. Then it gets the node with the name in inps[1] from the graph,
     * adds the node with the name in this TerminalSymbol to it as neighbor, and then
     * returns the graph.
     * 
     * |_ parser.TerminalSymbol: Bathroom
     */
    private static class ProcessNeighbor extends AGramSymThrowsVisitor</*R=*/Graph, /*P=*/Object> {
        public static final ProcessNeighbor Singleton = new ProcessNeighbor();
        
        private ProcessNeighbor() {
            // TODO: 2a
            // --------------------------------------------------------------------
            setCmd("Id", new IGramSymVisitorCmd<Graph,Object>() {
                @SuppressWarnings("unchecked")
                public Graph apply(String idx, IGrammarSymbol host, Object... inps) {
                    Graph g = (Graph)inps[0];
                    String edgeStartNodeName = (String)inps[1];
                    String nodeName = host.toString();
                    Node n = g.getNodes().get(nodeName);
                    if (n==null) {
                        // add node if it didn't already exist
                        g.getNodes().put(nodeName, n = new Node(nodeName));
                    }
                    Node start = g.getNodes().get(edgeStartNodeName);
                    start.getNeighbors().add(n);
                    return g;
                }
            });
            // --------------------------------------------------------------------
        }
    }
    
    /** Command that just returns inps[0]. */
    private static final IGramSymVisitorCmd<Graph,Graph> NOP = new IGramSymVisitorCmd<Graph,Graph>() {
        public Graph apply(String idx, IGrammarSymbol host, Graph... inps) {
            return inps[0];
        }
    };
    /** Command that just returns inps[0] as Graph. */
    private static final IGramSymVisitorCmd<Graph,Object> NOP_POBJECT = new IGramSymVisitorCmd<Graph,Object>() {
        @SuppressWarnings("unchecked")
        public Graph apply(String idx, IGrammarSymbol host, Object... inps) {
            return (Graph)inps[0];
        }
    };
}
