package token.tokenizer;

import token.*;
import java.io.*;
import java.util.*;


/**
 * Use the StreamTokenizer provided by in java.io to scan an input stream and extract an appropriate Token.
 */
public abstract class ATokenizer implements ITokenizer {
  /**
   * StreamTokenizer to use.
   */
  protected StreamTokenizer _st;
  
  /**
   * Reader to use.
   */
  protected Reader _fileReader;
  
  /**
   * Factory for the tokens
   */
  protected ITokenFactory tokFac = TokenFactory.Singleton;
  
  /**
   * One element buffer for the last token generated
   */
  protected Token _lastToken = tokFac.makeToken("EOF","EOF");;

  /**
   * Abstract command to get the next available token
   */
  private interface IGetTokenCmd {
    Token apply();
  }
  
  /**
   * Concrete command to get the next token from the StreamTokenizer
   * Saves the token as the _lastToken
   */
  private IGetTokenCmd _normalGet = new IGetTokenCmd(){
    public Token apply() {
      _lastToken = makeNextToken();
      return _lastToken;
    }
  };
  
  /**
   * Concrete command to return the last token rather that the next
   * token from the StreamTokenizer.   This is used when the last token 
   * has been pushed back into the tokenizer.
   * Resets the command to use the normal gettor process since the 
   * token buffer is only one token long.
   */
  private IGetTokenCmd _pushBackGet = new IGetTokenCmd(){
    public Token apply() {
      _getTokenCmd = _normalGet;
      return _lastToken;
    }
  };
  
  /**
   * The current command to use to get the next token
   */
  private IGetTokenCmd _getTokenCmd = _normalGet;
  
  /**
   * Initialize _st to read from a input Reader file with the given input file name.
   *
   * @param inputFileName the name of the input text file
   * @throws FileNotFoundException
   */
  public ATokenizer(String inputFileName) throws FileNotFoundException {
    // Create an input text file object:
    _fileReader = new FileReader(inputFileName);
    
    // Create a Streamtokenizer to parse the input text file:
    _st = new StreamTokenizer(_fileReader);
    
    _st.eolIsSignificant(false);
    
    // Where do we close the file? Java has no destructors!
    // DXN: close it in getNextToken() when the end-of-file is reached.
    
  }
  
  
  /**
   * Return the next token, either from the stream or the last token if it
   * was pushed back
   */
  public Token getNextToken() { 
    return _getTokenCmd.apply();
  }
  
  /**
   * Use _st to scan the input, extracts, and returns an appropriate concrete Token.
   *
   * @return next token
   * @throws IllegalArgumentException Thrown if an illegal input is encountered.
   */
  protected abstract Token makeNextToken();

  
  /**
   * Put the previously consumed token back into the token stream. Can only put back one token.
   */
  public void putBack() {
    _getTokenCmd = _pushBackGet; // Change the state of the tokenizer
  }
}



  
  
