package model;

import extvisitor.*;
import rac.*;
import listFW.*;
import listFW.visitor.*;

/**
 * Abstract search for a path in a graph.
 */
public class GraphSearch {
    /** Factory for the restricted access container that controls the visiting order. */
    protected IRACFactory<Node> _racFactory;
    /** List factory for the list that represents the path. */
    protected IListFactory<Node> _listFactory;
    
    /** Create a new Search instance using the two factories.
      * @param f RAC factory
      * @param lf list factory
      */
    public GraphSearch(IRACFactory<Node> f, IListFactory<Node> lf) {
        _racFactory = f;
        _listFactory = lf;
    }
    
    /** Search for a path from one node to the other. Returns an empty path if no
      * path is found.
      * @param g graph
      * @param start start node
      * @param end end node
      * @return list representing the path, or empty if no path found */
    @SuppressWarnings("unchecked")
    public IList<Node> search(Graph g, Node start, final Node end) {
        IRAContainer<Node> rac = _racFactory.makeRAC();
        SearchVisitor sv = new SearchVisitor(g, start, end, rac);
        return start.execute(sv).execute(new Reverse<Node>(), _listFactory);
        // the path comes out of the SearchVisitor in reverse, so reverse it
    }
    
    /** Extended visitor that performs the search. */
    protected class SearchVisitor extends GraphVisitor<IList<Node>,Void,IList<Node>> {
        /** End node, so we know when we have reached our goal. */
        protected Node _end;
        /** RAC controlling the nodes we should visit. */
        protected IRAContainer<Node> _rac;
        
        /** Create a new visitor with the specified start and end nodes and the RAC.
          * @param g graph
          * @param s start node
          * @param e end node
          * @param r RAC to use */
        public SearchVisitor(Graph g, Node s, Node e, IRAContainer<Node> r) {
            super(_listFactory.makeEmptyList());
            _end = e;
            _rac = r;
            // set the command for all nodes to Unvisited with an empty list as path
            for (Node n: g.getNodes().values()) {
                setCmd(n.getName(), new Unvisited(_listFactory.makeEmptyList()));
            }
        }
        
        /** Process the next node from the RAC. */
        public IList<Node> processNext() {
            // 1. execute a visitor on the RAC
            // 1.a. if the RAC is empty, return the empty list (no path was found)
            // 1.b. if the RAC is non-empty...
            // 1.b.1. get the next node from the RAC
            // 1.b.2. execute the SearchVisitor on the node
            // 1.b.3. the return value of the SearchVisitor is a list of nodes.
            //        if it is empty, no path was found; if it is non-empty, a
            //        path was found.
            //        execute a visitor on the return value
            // 1.b.3.a. if the returned list is empty, process the next node from the RAC
            // 1.b.3.b. if the returned list is non-empty, return that list
            // --------------------------------------------------------------------
            // TODO: 3a

            // --------------------------------------------------------------------
        }
        
        /** Command for visited nodes. */
        protected class Visited extends NodeCommand<IList<Node>,Void,IList<Node>> {
            /** Create a new Visited command with the specified path.
              * @param p path to this node */
            public Visited(IList<Node> p) { super(p); }
            /** Process this node, i.e. take the next node out of the RAC. */
            public IList<Node> apply(String idx, Node host, Void... params) {
                // this node been visited
                // 1. process the next node from the RAC
                // --------------------------------------------------------------------
                // TODO: 3a

                // --------------------------------------------------------------------
            }
            
        }

        /** Command for unvisited nodes. */
        public class Unvisited extends NodeCommand<IList<Node>,Void,IList<Node>> {
            /** Create a new Visited command with the specified path.
              * @param p path to this node */
            public Unvisited(IList<Node> p) { super(p); }
            /** Process this node. */
            @SuppressWarnings("unchecked")
            public IList<Node>apply(String idx, Node host, Void... params) {
                // this node has not been visited yet
                // 1. append this node to the path (use getState() and setState())
                // 2. set the command for this node to Visited(getState())
                // 3. if we have reached the end, return the path (use getState())
                // 4. process all neighbors of the node:
                // 4.1. get the command for the neighbor's node
                // 4.2. set the path for the neighbors node to this node's path
                // 4.3. set the updated command for the neighbor's node
                // 4.4. add the neighbor's node to the RAC
                // 5. process the next node from the RAC
                // --------------------------------------------------------------------
                // TODO: 3a

                // --------------------------------------------------------------------
            }
        }
    }
}
