package farmer;

import graphs.*;
import lrs.*;
import lrs.visitor.*;
import java.util.*;
import fp.*;

/**
 * A wrapper class that wraps a State in an IVertex
 * so that it will be compatible with the graph traversal
 * algorithms.   Also defines situation specific entities
 * such as the ending state and the list of ok next states.
 */
public class FarmerVertex implements IVertex {
  private State _endState;  // defines the last, sink state.
  
  private State _currentState; // the state that this vertex represents.
  
  private LRStruct _okSets; // the list of all valid next states.
  
  private LRStruct _nextStates; // the valid next states from the current state
  private int _nNhbrs;
  
  /**
   * Constructor for the class.
   * If the current state is the end state, then this vertex will have 
   * no neighbors (i.e. will be a sink).
   */ 
  public FarmerVertex(State currentState, State endState, LRStruct okSets) {
    _currentState = currentState;
    _endState = endState;
    _okSets = okSets;
    
    if(0==_currentState.compareTo(_endState)) {
      _nextStates = new LRStruct();  // end state is a sink
      _nNhbrs = 0;
    }
    else {
      _nextStates = _currentState.makeOkNextStates(_okSets);
      _nNhbrs = (Integer) _nextStates.execute(LengthLRS.Singleton, null);
    }
  }
  
  
  /**
   * The current state is the data being held
   */
  public Object getDat() {
    return _currentState;
  }
  
  /**
   * Lazily evaluate the neighboring vertices.
   * If there are no neighbors (i.e. next states) 
   * then don't calculate any neighbors.
   */
  public Set<IVertex> getNhbrs() {
    final Set<IVertex> nhbrs = new HashSet<IVertex>();
    
    _nextStates.execute(new MapLRS( new ILambda() {
      public Object apply(Object...args) {
        nhbrs.add(new FarmerVertex((State)((LRStruct)args[0]).getFirst(),
                                   _endState,
                                   _okSets));
        return true;
      }
    }), null);
    
    return nhbrs;
  }
  
  public Object execute(IVertexAlgo algo, Object... params) {
    return algo.caseAt(_nNhbrs, this, params);
  }
  
  /**
   * Needed for use in Hashtables
   * Used in conjunction with hashCode()
   * This is needed because we want different instances of FarmerVertex
   * holding the same current state to behave as if they were equal.
   */
  public boolean equals(Object other) {
    if(other instanceof FarmerVertex) {
      return 0==_currentState.compareTo(((FarmerVertex)other)._currentState);
    }
    else return false;
  }
  
  /**
   * Needed for use in Hashtables in conjunction with equals()
   */
  public int hashCode() {
    return 0;  // Very ineffecient.  Required for use in Hashtables.
  }
  
  public String toString() { 
    return _currentState.toString();
  }
}

