package graphs.visitor;
import graphs.*;
import graphs.rac.*;
import fp.*;
import java.util.*;

/**
 * Abstract RAC-based traversal of an IVertex using forward accumulation
 * Handles loops.
 * Takes 3 input lambda:
 * fns[0] = initial vertex (base #1) case.  
 *          Takes the data element as input.
 * fns[1] = non-zero links (inductive) case.  
 *          Takes the data plus the accumulated result. 
 *          Returns the new accumulated result.
 * fns[2] = Loop case (base #2) case.  
 *          Used when IVertex object has been seen already.  
 *          Input is data element and recursive result.
 *          Returns the new accumulated result.
 */
public class RACTravAlgo implements IVertexAlgo {
  
  private Set<IVertex> seen = new HashSet<IVertex>();
  
  private ARAC<IVertex> rac;
  
  public RACTravAlgo(ARAC<IVertex> rac) {
    this.rac = rac;
  }
  
  public Object caseAt(int nNbhrs, IVertex host, Object...fns){
    // don't need check seen here b/c this is first time and no recur through here.
    rac.storeAll(host.getNhbrs()); // add neihbors, if any

    seen.add(host);
    Object result = processRAC(((ILambda) fns[0]).apply(host.getDat()), fns);
    seen.remove(host);            
    return result;    
  }   
  
  /**
   * This is the actual recursive method that processes the RAC
   * @param acc  the accumulated result so far
   * @param fns The ILambdas used to process the data in the various cases.
   * @return The result of the traversal.
   */
  private Object processRAC(Object acc, Object... fns) {
    if(rac.isEmpty()) {
      return acc;  // Done!
    }
    else {
      Object result;
      IVertex g = rac.retrieve();
      if(seen.contains(g)) {
        acc = ((ILambda) fns[2]).apply(g.getDat(), acc);
        result = processRAC(acc, fns);// keep going.
      }
      else {
        rac.storeAll(g.getNhbrs()); // add neighbors, if any
        acc = ((ILambda) fns[1]).apply(g.getDat(), acc);

        seen.add(g); // mark this vertex
        result = processRAC(acc, fns); // keep going
        seen.remove(g); // unmark this vertex
      }
      return result;
    }
  }
}
