package treap;

import brs.visitor.*;
import brs.*;
import java.util.*;
import fp.*;

/**
 * Inserts an Object into the host maintaining the host's binary search tree
 * property via a given Comparator.
 * Duplication is not allowed: replaces old data object with the new one.
 * @author Dung X. Nguyen - Copyright 2003 - All rights reserved.
 */
public class TreapInserter_SOL implements IVisitor {
    private Comparator<TreapTriple> _orderKey;
    private Comparator<TreapTriple> _orderPriority;
    private IVisitor _remove;
    
    private IVisitor _insertHelp = new IVisitor() {
        /**
         * Returns the host tree where the input is inserted as the root.
         * @param host an empty binary tree, which obviously satisfies BSTP.
         * @param input new data to be inserted.
         * @return BiTree host, which is no longer empty.
         */
        public Object emptyCase(BiTree host, Object input) {
            host.insertRoot (input);
            return host;
        }
        
        /**
         * Inserts the input object, assuming the input is NOT in the host tree.
         * @param host non-empty and satisfies Treap property.
         * @param input new data to be inserted.
         * @return BiTree the host tree.
         */
        public Object nonEmptyCase(BiTree host, Object input) {
            TreapTriple inp = (TreapTriple) input;        
            TreapTriple root = (TreapTriple) host.getRootDat();
            if (_orderKey.compare(inp, root) < 0) {
                host.getLeftSubTree().execute(this, input); 
                if (_orderPriority.compare(root, (TreapTriple)host.getLeftSubTree().getRootDat()) > 0) {                
                    host.execute(RotateRight.Singleton, null);
                }
            }
            else if (_orderKey.compare(inp, root) > 0) {
                host.getRightSubTree().execute(this, input);
                if (_orderPriority.compare(root, (TreapTriple) host.getRightSubTree().getRootDat()) > 0) {
                    host.execute(RotateLeft.Singleton, null);
                }
            }
            else {
                host.setRootDat(inp);
            }
            return host;
        }    
    };
       
    /**
     * Used when the items in the tree are Comparable objects.
     */
    public TreapInserter_SOL() {
        _orderKey = new Comparator<TreapTriple>() {
            public int compare(TreapTriple x, TreapTriple y) {
                return x.getKeyComparable().compareTo(y);
            }
        };
        _orderPriority = new Comparator<TreapTriple>() {
            public int compare(TreapTriple x, TreapTriple y) {
                return x.getPriorityComparable().compareTo(y);
            }
        };
        _remove = new TreapDeleter_SOL();
    }
    
    public  TreapInserter_SOL (Comparator<TreapTriple> orderKey,
                            Comparator<TreapTriple> orderPriority) {
        _orderKey = orderKey;
        _orderPriority = orderPriority;
        _remove = new TreapDeleter_SOL(_orderKey, _orderPriority);
    }
    
    /**
     * Returns the host tree where the input is inserted as the root.
     * @param host an empty binary tree, which obviously satisfies BSTP.
     * @param input new data to be inserted.
     * @return BiTree host, which is no longer empty.
     */
    public Object emptyCase(BiTree host, Object input) {
        host.insertRoot (input);
        return host;
    }
    
    /**
     * If the input is equal to host.getRootDat) then the old root data is
     * replaced by input.  Equality here is implicitly defined by the total
     * orderKeying represented by the Comparator _orderKey.
     * @param host non-empty and satisfies BSTP.
     * @param input new data to be inserted.
     * @return BiTree the host tree.
     */
    public Object nonEmptyCase(BiTree host, Object input) {
        host.execute(_remove, input);
        return host.execute(_insertHelp, input);
    }
}

