package heap;

import java.util.*;

/**
 * Maintains the heap structure of a given array of Objects.
 * Uses a stored Comparator to determine a binary ordering of the objects.
 * An array A[lo:hi] is said to satisfy the "heap property" if
 * for i = 0,.., hi - lo:
 * A[lo + i] <= A[lo + 2 * i + 1]  and
 * A[lo + i] < = A[lo + 2 * i + 2],
 * whenever these indices are in the range [lo..hi].
 * Copyright 2005 - Dung X. Nguyen and Stephen Wong - All rights reserved. </p>
 */
public class Heapifier {

  private Comparator comp;
  
  public Heapifier(Comparator comp) {
    this.comp = comp;
  }
  
  /**
   * "Sifts" A[cur] up the array A to maintain the heap property.
   * @param A A[lo:cur-1] is a heap.
   * @param lo the low index of A.
   * @param cur lo <= cur <= the high index of A.
   */
  public void siftUp(Object[] A, int lo, int cur) {
    Object dat = A[cur];
    int parent = (cur - lo - 1) / 2 + lo;  // index of parent.
    while (0 < (cur - lo) && comp.compare(dat,A[parent]) < 0) {
      A[cur] = A[parent];
      cur = parent;
      parent = (cur - lo - 1) / 2 + lo;
    }
    A[cur] = dat;
  }
  
  
  /**
   * "Sifts" the value at A[cur] down the array A to maintain the heap property.
   * @param A  A[cur + 1:hi] satisfies the heap property.
   * @param lo the low index of A.
   * @param cur lo <= cur <= hi.
   * @param hi  the high index of a.
   */
  public void siftDown(Object[] A, int lo, int cur, int hi) {
    Object dat = A[cur]; // hold on to data.
    int child = 2 * cur + 1 - lo; // index of left child of A[cur].
    boolean done = hi < child;
    while (!done) {
      if (child < hi && comp.compare(A[child + 1],A[child]) < 0) {
        child++;
      } // child is the index of the smaller of the two children.
      if (comp.compare(A[child], dat) < 0) {
        A[cur] = A[child];
        cur = child;
        child = 2 * cur + 1 - lo;
        done = hi < child;
      }  // parent of A[cur] < dat.
      else {// dat <= A[child], the smaller of the children of A[cur].
        done = true;  // heap condition is satisfied.
      } // A[cur] is less than its children.
    } // location found for dat.
    A[cur] = dat;
  }
}  