package parser;
import genvisitor.*;

/**
 * A factory to create visitors to parse a binary sequence of grammar tokens.
 * To easily create a sequence of two or more grammar tokens, use 
 * MultiSequenceFact rather than multiple SequenceFacts.
 */
public class SequenceFact extends ATVFactory {
  
   /**
   * The name of this instance of the factory
   */
  private String _name;
  
  /**
   * Factory for the first grammar non-terminals.
   */
  private ITVFactory _fact1;
  
  /**
   * Visitor for the first grammar non-terminals.
   */
  private ATokVisitor<IGrammarToken, Object> _parse1;
  
  /**
   * Factory for the second grammar non-terminals.
   */
  private ITVFactory _fact2;
  
  /**
   * Visitor for the second grammar non-terminals.
   */
  private ATokVisitor<IGrammarToken, Object> _parse2;
  
  /**
   * Initializer lambda for this factory.
   */
  private ILambda _initializer = new ILambda() {
    public Object apply(Object param) {
      // change state to no-op
      _initializer = NoOpLambda.Singleton;
      
      // initialize
      _parse1 = _fact1.makeVisitor();
      _parse2 = _fact2.makeVisitor();
      return null;
    }
  };
  
  /**
   * Constructor for the sequence factory,
   *
   * @param name The grammar token's name of this instance of a sequence.
   * @param tkz    tokenizer to use
   * @param fact1  factory for the first non-terminals
   * @param fact2 factory for the second non-terminals
   */
  public SequenceFact(String name, ITokenizer tkz, ITVFactory fact1, ITVFactory fact2) {
    super(tkz);
    _name = name;
    _fact1 = fact1;
    _fact2 = fact2;
  }
  
  
  /**
   * Make a token visitor to parse an sequence non-terminal.
   *
   * @return token visitor
   */
  public ATokVisitor<IGrammarToken, Object> makeVisitor() {
    initialize();
    
    return new ATokVisitor<IGrammarToken, Object>(_parse1)
    {
      {
        // Decorate every command that was copied from _parse1
        map(new AGenVisitor.IMapLambda<String, AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken>>() {
          public AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken> apply(String idx, final AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken> cmd) {
            return new AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken>() {
              public IGrammarToken apply(String idx, AToken host, Object... inps) {
                System.out.println("Sequence: "+_name);
                return new SequenceToken(_name, (IGrammarToken)cmd.apply(idx, host, inps), (IGrammarToken) nextToken().execute(_parse2, inps));
              }
            };
          }
        });
        final AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken> oldDefaultCmd = getDefaultCmd();
        setDefaultCmd(new AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken>() {
          public IGrammarToken apply(String idx, AToken host, Object... inps) {
            return new SequenceToken(_name,(IGrammarToken)oldDefaultCmd.apply(idx, host, inps), (IGrammarToken) nextToken().execute(_parse2, inps));
          }
        });
      }
    };
  }
  
  /**
   * Make a token visitor that delegates to the given visitor in a chain of responsibility
   *
   * @param successor visitor to serve as successor in the chain
   */
  public ATokVisitor<IGrammarToken, Object> makeChainedVisitor(ATokVisitor<IGrammarToken, Object> successor) {
    initialize();
    
    final ATokVisitor<IGrammarToken, Object> v = makeVisitor();
    // Copy all cmds from successor to v
    successor.map(new AGenVisitor.IMapLambda<String, AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken>>() {
      public AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken> apply(String idx, final AGenVisitor.IGenVisitorCmd<IGrammarToken, String, Object, AToken> cmd) {
        v.addCmd(idx, cmd);   // copy command to v
        return cmd;  // put cmd back in successor
      }
    });
    v.setDefaultCmd(successor.getDefaultCmd());
    return v;
  }
  
  /**
   * Make the visitor.
   */
  private void initialize() {
    _initializer.apply(null);
  }
}

