package genvisitor;

import java.util.*;

/**
 * Hash-table based implementation of IGenVisitor.
 * Cases are defined by loading IGenVisitorCmd objects keyed to index values 
 * that identify different hosts.
 * Case-dependent behavior is guaranteed through the supplied default case. 
 * @param R The return type
 * @param I The index type used by the host to select a case
 * @param P The vararg parameter type
 * @param H The type of the host this visitor expects to see.
 */
public class AGenVisitor<R, I, P, H extends IGenVisitorHost<I, H>> implements IGenVisitor<R, I, P, H> {
  
  /**
   * A command that corresponds to a particular case.
   */
  public interface IGenVisitorCmd<R, I, P, H extends IGenVisitorHost<I, H>> {
    public R apply(I idx, H host, P... inps);
  }
  
  /**
   * The mapping from index value to corresponding command object.
   */
  private Map<I, IGenVisitorCmd<R, I, P, H>> cmds = new Hashtable<I, IGenVisitorCmd<R, I, P, H>>();
  
  /**
   * The default command to use for all unmapped cases.
   */
  private IGenVisitorCmd<R, I, P, H> defaultCmd;
  
  /**
   * The constructor for the class.
   * 
   * @param defaultCmd  The default command to use for any unmapped caes.
   */
  public AGenVisitor(IGenVisitorCmd<R, I, P, H> defaultCmd) {
    this.defaultCmd = defaultCmd;
  }
  
  /**
   * Copy constructor.  Instantiates a visitor that shares the same commands 
   * as the given visitor.
   * @param other The visitor to copy from.
   */
  public AGenVisitor(AGenVisitor<R, I, P, H> other) {
    this(other.defaultCmd);
    for( I idx: other.cmds.keySet()) {
      cmds.put(idx, other.cmds.get(idx));
    }    
  }
  
  /**
   * The parameterized case statement called by a host's execute method.
   * The host's execute method calls caseAt with an index value 
   * corresponding to that host.   The command that is stored that corresponds to 
   * the supplied index key is executed.   If no command is mapped to the key,
   * the default command is executed.
   * @param idx The index supplied by the host
   * @param host The host itself
   * @param inps The vararg input parameters
   */
  public R caseAt(I idx, H host, P... inps)  {
    IGenVisitorCmd<R, I, P, H> cmd = cmds.get(idx);
    if(null == cmd) return defaultCmd.apply(idx, host, inps);
    else return cmd.apply(idx, host, inps);
  }
  

  /**
   * Replaces the existing default command
   * @param defaultCmd  The new default command
   */
  public void setDefaultCmd(IGenVisitorCmd<R, I, P, H> defaultCmd) {
    this.defaultCmd = defaultCmd;
  }

  /**
   * Returns the existing default command
   * @return  The current default command
   */
  public IGenVisitorCmd<R, I, P, H> getDefaultCmd() {
    return defaultCmd;
  }

  /**
   * Returns the command keyed to an index value.  Throws an exception if the 
   * key doesn't exist.
   * @param idx The index key value
   * @return The keyed commmand
   */
  public IGenVisitorCmd<R, I, P, H>  getCmd(I idx) {
    return cmds.get(idx);
  }

  /**
   * Adds a command keyed to an index value.  Replaces any commands already 
   * keyed to the supplied index value.
   * @param idx The index key value
   * @param cmd The command to save
   */
  public void addCmd(I idx, IGenVisitorCmd<R, I, P, H> cmd) {
    cmds.put(idx, cmd);
  }
  
  /**
   * Lambda used for mapping over the stored commands
   */
  public interface IMapLambda<I, V> {
    public V apply(I idx, V oldCmd);
  }
  
  /**
   * Map the given lambda over the stored commands.  
   * The return value of the lambda replaces the existing command.
   * This map function can thus be used to copy or replace all the
   * stored commands without knowing what they are.
   * @param lambda  The IMapLambda to map across the commands.
   */
  public void map(IMapLambda<I, IGenVisitorCmd<R, I, P, H>> lambda) {
    for( I idx: cmds.keySet()) {
      cmds.put(idx, lambda.apply(idx, cmds.get(idx)));
    }
  }
  
}