  package parser;
import genvisitor.*;

/**
 * Utility factory that automatically chains together a series of 
 * SequenceFactories.  The result is to create a single factory that 
 * appears to create an arbitrarily long sequence of ITVFactories.
 * Note that each SequenceToken generated in the chain is given a 
 * unique, numbered name to aid in debugging.  The first (returned) one
 * has the given name.
 * SequenceTokens only ever call the "Sequence" case of their visitors,
 * so the individual tokens are indistinguishable to the visitors.
 */
public class MultiSequenceFact implements ITVFactory {
  private ITVFactory _composed;
  
  /**
   * Constructor for the sequence factory,
   *
   * @param name The grammar token's name of this instance of a sequence.  Individual sequence names are made by appending a numerical value to this name.
   * @param tkz    tokenizer to use
   * @param fact1  factory for the first non-terminals
   * @param fact2 factory for the second non-terminals
   * @param facts a vararg array of additional factories of non-terminals
   */
   public MultiSequenceFact(String name, ITokenizer tkz, ITVFactory fact1, ITVFactory fact2, ITVFactory...facts) {

     if(0 == facts.length) {
       _composed = makeComposed(name, tkz, fact1, fact2);
     }
     else {
       ITVFactory temp = facts[facts.length-1];
       for(int i = facts.length-2; i >= 0 ; i--) {
         temp = makeComposed(name +(i+2), tkz, facts[i], temp);
       }
       temp = makeComposed(name+1, tkz, fact2, temp);
       _composed = makeComposed(name, tkz, fact1, temp);
     }
   }
 
   private ITVFactory makeComposed(String s, ITokenizer t, ITVFactory fact1, ITVFactory fact2) {
     return new SequenceFact(s, t, fact1, fact2);
   }
   
   
  public ATokVisitor<IGrammarToken, Object> makeVisitor() {
    return _composed.makeVisitor();
  }
  
  public ATokVisitor<IGrammarToken, Object> makeChainedVisitor(ATokVisitor<IGrammarToken, Object> successor) {
    return _composed.makeChainedVisitor(successor);
  }  
   
}