package dijkstra;

import rac.*;
import java.util.*;
import heap.*;
import listFW.*;
import java.lang.reflect.Array;

/**
 * An IRACFactory that returns a priority queue for use in Dijkstra's
 * shortest path algorithm.   The ordering of 
 * the queue is determined by a given Comparator object or a default 
 * Comparator that returns the "largest" Comparable element first.
 * Internally, uses a heap to store the data.
 */
public class DijkstraPQueueFactory extends AArrayRACFactory<INEList<VertexDistDyad>> {
  
  private int maxSize = 7;  // must be 2^n -1.  Set small here for easier testing.
  
  private Comparator<INEList<VertexDistDyad>> comp =new Comparator<INEList<VertexDistDyad>>() {
    public int compare(INEList<VertexDistDyad> x, INEList<VertexDistDyad> y) {
      if(x.getFirst().getDistance() == y.getFirst().getDistance()) return 0;
      else return (x.getFirst().getDistance() > y.getFirst().getDistance()) ? +1 : -1;
    }
  };
  
  /**
   * Used when the items in the container are Comparable objects.
   * The "largest" element has the highest priority.
   */
  public DijkstraPQueueFactory() {
  }
  
  /**
   * Used when we want to prioritize the items according to a given Comparator.
   * @param comp The item that is smallest according to comp has the highest
   * priority. 
   */
  public DijkstraPQueueFactory(Comparator<INEList<VertexDistDyad>> comp) {
    this.comp = comp;
  }
  
  /**
   * Factory method for the RAC
   */
  public IRAContainer makeRAC() {
    // Work-around for lack of generic creation:
    // "new INEList<VertexDistDyad>[maxSize]" = "(INEList<VertexDistDyad>[])Array.newInstance(INEList.class,maxSize)" 
    return new AArrayContainer<INEList<VertexDistDyad>>((INEList<VertexDistDyad>[])Array.newInstance(INEList.class,maxSize)){ 
      
      /**
       * Location in the heap of a given Vertex
       */
      private Map<Vertex, Integer> heapLoc = new HashMap<Vertex, Integer>();
      
      /**
       * Set of vertices for which the minimum distance has already been found
       */
      private Set<Vertex> doneSet = new HashSet<Vertex>();
     
      /**
       * Remove the next item from the container and return it.
       * Internally, retains a reference to the Vertex of first dyad 
       * every item (list of VertexDistDyads) ever returned.
       * @throw an IndexOutOfBoundsException if this IRAContainer is empty.
       */
      public INEList<VertexDistDyad> get() {
        // STUDENT TO MODIFY
        if(end <= 0) throw new IndexOutOfBoundsException("The RAC is empty.");
        INEList<VertexDistDyad> result = data[0];
        data[0] = data[end-1];
        end--;
        siftDown(data, 0, 0, end-1);
        return result;
      }
      
      
      /**
       * Add an item to the container.
       * Will not store item if the Vertex of the first VertexDistDyad in 
       * the input list previously returned by get();
       * If input.getFirst().getVertex() (the first Vertex in the list)
       * already exists in the data store, then keep only the one
       * with the smallest distance.
       * @param input the Object to be added to this IRAContainer.
       * @throw an IndexOutOfBOundsException if this IRAContainer is full.
       */
      public void put(INEList<VertexDistDyad> input) {
        
        VertexDistDyad inputDyad = input.getFirst();
        // STUDENT TO MODIFY
        
        if(isFull()) throw new IndexOutOfBoundsException("RAC is full.");
        
        data[end++] = input;
        siftUp(data, 0, end);
        return;
      }
      
      /**
       * Modified siftUp routine that records any change in 
       * a data item's position in the heap in a map structure.
       * The map structure maps the Vertex of the first VertexDistDyad
       * in a data item to the data item's location in the heap.
       * @param A A[lo:cur-1] is a heap.
       * @param lo the low index of A.
       * @param cur lo <= cur <= the high index of A.
       */
      private void siftUp(INEList<VertexDistDyad>[] A, int lo, int cur) {
        // STUDENT TO MODIFY
        INEList<VertexDistDyad> dat = A[cur];
        int parent = (cur - lo - 1) / 2 + lo;  // index of parent.
        while (0 < (cur - lo) && comp.compare(dat,A[parent]) <= 0) {
          A[cur] = A[parent];
          cur = parent;         
          parent = (cur - lo - 1) / 2 + lo; // next parent
        }
        A[cur] = dat;
      }
      
      /**
       * Modified siftDown routine that records any change in 
       * a data item's position in the heap in a map structure.
       * The map structure maps the Vertex of the first VertexDistDyad
       * in a data item to the data item's location in the heap.
       * @param A  A[cur + 1:hi] satisfies the heap property.
       * @param lo the low index of A.
       * @param cur lo <= cur <= hi.
       * @param hi  the high index of A (note that it is inclusive!).
       */
      private void siftDown(INEList<VertexDistDyad>[] A, int lo, int cur, int hi) {
        // STUDENT TO MODIFY
        INEList<VertexDistDyad> dat = A[cur]; // hold on to data.
        int child = 2 * cur + 1 - lo; // index of left child of A[cur].
        boolean done = hi < child;
        while (!done) {
          if (child < hi && comp.compare(A[child + 1],A[child]) < 0) {
            child++;
          } // child is the index of the smaller of the two children.
          if (comp.compare(A[child], dat) < 0) {
            A[cur] = A[child];
            cur = child;
            child = 2 * cur + 1 - lo;
            done = hi < child;
          }  // parent of A[cur] < dat.
          else {// dat <= A[child], the smaller of the children of A[cur].
            done = true;  // heap condition is satisfied.
          } // A[cur] is less than its children.
        } // location found for dat.
        A[cur] = dat;
      }
      
      /**
       * Return the next element in this IRAContainer withour removing it.
       * @throw an IndexOutOfBoundsException if this IRAContainer is empty.
       */     
      public INEList<VertexDistDyad> peek() {
        if(end <= 0) throw new IndexOutOfBoundsException("The RAC is empty.");
        return data[0];
      }
      
    };
  }
}