package view;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * Serves as a view to Hangman.
 * Has a JPanel where the body parts are painted.
 * Uses a JApplet to supply the Graphics context for the JPamel to draw the body
 * parts.
 * To paint something in Java, we need to paint on a JPanel and override
 * its paintComponent(...) method.
 * Everytime the JPanel needs repainting, it will call paintComponent(...).
 * For example, when the JFrame or a JApplet that contains the JPanel is
 * resized, it will try to update all of its contents, in particular it will
 * call its JPanel to update, which in turns will schedule a call to
 * repaint(...), which in turn will call paintComponent(...)!
 */
public class HangmanGUI {

    /**
     * The GUI component that holds all other GUI components.
     */
    private JApplet _applet;

    /**
     * The adapter used to communicate with the model to tell it to paint the
     * body parts.
     */
    private IPaintAdapter _paintAdapter;

    /**
     * A panel derived from a JPanel where the body parts are drawn.
     * The JPanel's paint() method is overriden so that the gallows is
     * automatically drawn when the panel is repainted.
     */
    private JPanel _displayPanel = new JPanel() {
        public void paintComponent(Graphics g) { // called whenever the panel is
            // repainted.  The graphics context g is supplied by the GUI
            // component that contains this JPanel.
            super.paintComponent(g);   // do whatever usually is done,
                // e.g. clear the panel.
            g.setColor(Color.black);  // set the drawing color
            g.fillRect(0,220,200,5);  // base of scaffold
            g.fillRect(0,0,70,5);     // top of scaffold
            g.fillRect(0,0,5,220);    // side of scaffold

            // FOR ILLUSTRATION ONLY:
            System.out.println("HangmanGUI._displayPanel.paintComponent() " +
                "calling _paintAdapter.paint()...");

            //  Delegate to the adapter to get the body parts drawn:
            _paintAdapter.paint(g);
        }
    };


    /**
     * Initializes the GUI components.
     * @param a the applet that holds all GUI components.
     * @param gc The IGameControlAdapter object used to for guessing and
     * resetting.
     * @param pa The IPaintAdapter object used for requesting that the model
     * paint the body parts.
     */
    public HangmanGUI(JApplet a, IPaintAdapter pa) {
        if (null == a || null == pa) {
            throw new IllegalArgumentException("HangmanGUI(...) has null " +
                "arguments!");
        }
        // FOR ILLUSTRATION ONLY:
        System.out.println("HangmanGUI.constructor initializing _displayPanel "
            + "and _applet...");

        _applet = a;
        _paintAdapter = pa;
        guiInit();
    }

    /**Component initialization*/
    private void guiInit() {
        _displayPanel.setBackground(Color.white);
        _applet.setSize(400, 350);
        Container contentPane = _applet.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(_displayPanel, BorderLayout.CENTER);
    }
}