package view.console;

import model.AUnit;
import model.Measurement;
import view.IView;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.util.List;

/**
 * Console view of the temperature calculator.
 *
 * @author Mathias Ricken
 */
public class ConsoleView implements IView {   
    /**
     * List of units.
     */
    private List<AUnit> _units;
    
    /**
     * Buffered reader from standard in.
     */
    private BufferedReader _br = new BufferedReader(new InputStreamReader(System.in));
    
    /**
     * Constructs a new temperature calculator console view.
     *
     * @param units list of units
     */
    public ConsoleView(List<AUnit> units) {
        if (units.size()<2) {
            throw new IllegalArgumentException("Need at least two units.");
        }
        _units = units;
    }
    
    /**
     * Run the calculator.
     */
    public void run() {
        System.out.println("Convert temperature between arbitrary scales.");
        String s = null;
        do {
            System.out.println("Available units:");
            int i = 0;
            for(AUnit u: _units) {
                System.out.println("  "+String.valueOf(++i)+": "+u.toString());
            }
            i = integerChoice("Type '1' .. '"+String.valueOf(_units.size())+"' to convert from the respective scale, or '0' to add a scale.", 
                              0, _units.size()+1);
            if (i==0) {
                addUnit();
                s = "y";
                continue;
            }
            convertFrom(_units.get(i-1));
            System.out.println("Do you want to convert more temperatures? ");
            s = multipleChoice("Press 'y' for yes, 'n' for no.", "y", "n");
        } while(s.equals("y"));
    }
    
    /**
     * Ask user to pick one of multiple choices.
     * @param prompt prompt to display
     * @param choices varargs with different choices
     * @return selected choice
     */
    private String multipleChoice(String prompt, String... choices) {
        String s = null;
        boolean repeat = true;
        do {
            System.out.print(prompt+" ");
            try {
                s = _br.readLine();
            }
            catch(IOException ioe) {
                continue;
            }
            repeat = true;
            for(String c: choices) {
                if (c.equals(s)) {
                    repeat = false;
                    break;
                }
            }
        } while(repeat);
        return s;
    }
    
    /**
     * Ask user to pick an integer.
     * @param prompt prompt to display
     * @param min minimum number, inclusive
     * @param max maximum number, exclusive
     * @return selected choice
     */
    private int integerChoice(String prompt, int min, int max) {
        int i = 0;
        do {
            System.out.print(prompt+" ");
            try {
                i = Integer.valueOf(_br.readLine());
            }
            catch(NumberFormatException nfe) {
                continue;
            }
            catch(IOException ioe) {
                continue;
            }
        } while((i<min) || (i>=max));
        return i;
    }
    
    /**
     * Perform a temperature conversion from the selected scale.
     * @param source source scale
     */
    private void convertFrom(AUnit source) {
        System.out.println();
        
        double d;
        boolean repeat = true;
        do {
            System.out.print("Please enter temperature in "+source.toString()+": ");
            try {
                d = Double.valueOf(_br.readLine());
                Measurement m = new Measurement(d, source);
                StringBuilder sb = new StringBuilder();
                sb.append(m);
                for(AUnit dest: _units) {
                    if (dest==source) {
                        continue;
                    }
                    sb.append(" = ");
                    sb.append(dest.convertTo(m));
                }
                System.out.println(sb.toString());
                repeat = false;
            }
            catch(NumberFormatException nfe) {
                // nothing to do, just repeat
            }
            catch(IOException ioe) {
                // nothing to do, just repeat
            }
        } while(repeat);
    }
    
    /**
     * Add a user-specified unit.
     */
    private void addUnit() {
        System.out.println();

        boolean repeat = true;
        do {
            System.out.print("Enter the class name for the scale to be added: ");
            try {
                String className = _br.readLine();
                if (!className.equals("")) {
                    Class unitClass = Class.forName(className);
                    if (!AUnit.class.isAssignableFrom(unitClass)) {
                        continue;
                    }
                    Constructor unitCtor = unitClass.getConstructor();
                    AUnit unit = (AUnit)unitCtor.newInstance();
                    _units.add(unit);
                    repeat = false;
                }
            }
            catch(Exception ioe) {
                // nothing to do, just repeat
            }
        } while(repeat);        
    }
}
