package view;

import model.AUnit;
import model.IMeasurable;
import model.length.Meter;
import model.length.Yard;
import model.temperature.Celsius;
import model.temperature.Fahrenheit;
import model.temperature.Kelvin;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Constructor;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;

/**
 * View of the conversion calculator.
 *
 * @author Mathias Ricken
 */
public class ConvCalcView extends JPanel {
    /**
     * Combobox with different measurables.
     */
    private JComboBox _measurablesBox;
    
    /**
     * Listbox with different units.
     */
    private JList _unitList;
    
    /**
     * Constructs a new conversion calculator view.
     */
    public ConvCalcView() {
        try {
            jbInit();
        }
        catch(Exception e) {
            e.printStackTrace();
        }
    }
    
    /**
     * Initialize the GUI components.
     */
    private void jbInit() {
        setSize(new Dimension(540, 380));
        setLayout(new BorderLayout());
        JPanel measurablesPanel = new JPanel();
        measurablesPanel.setLayout(new BorderLayout());
        measurablesPanel.add(new JLabel("Select measurable:"), BorderLayout.NORTH);
        _measurablesBox = new JComboBox();
        _measurablesBox.addItem(new ConcreteMeasurableChoice("model.temperature.ITemperature", new AUnit[]{
            new Celsius(),
                new Fahrenheit(),
                new Kelvin()
        }));
            _measurablesBox.addItem(new ConcreteMeasurableChoice("model.length.ILength", new AUnit[]{
                new Meter(),
                    new Yard(),
            }));
                _measurablesBox.addItem(new AddMeasurableChoice());
                _measurablesBox.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        measurablesBoxSelected(e);
                    }
                });
                measurablesPanel.add(_measurablesBox, BorderLayout.CENTER);
                add(measurablesPanel, BorderLayout.NORTH);
                JPanel unitPanel = new JPanel();
                unitPanel.setLayout(new BorderLayout());
                unitPanel.add(new JLabel("Select units to convert:"), BorderLayout.NORTH);
                _unitList = new JList();
                JScrollPane sp = new JScrollPane(_unitList);
                unitPanel.add(sp, BorderLayout.CENTER);
                add(unitPanel, BorderLayout.CENTER);
                JPanel buttonPanel = new JPanel();
                buttonPanel.setLayout(new FlowLayout());
                JButton convertBtn = new JButton("Convert...");
                convertBtn.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        convertBtnClicked(e);
                    }
                });
                buttonPanel.add(convertBtn);
                JButton addBtn = new JButton("Add Unit...");
                addBtn.addActionListener(new ActionListener() {
                    public void actionPerformed(ActionEvent e) {
                        addBtnClicked(e);
                    }
                });
                buttonPanel.add(addBtn);
                add(buttonPanel, BorderLayout.SOUTH);
                ((IMeasurableChoice)_measurablesBox.getSelectedItem()).select();
    }
    
    /**
     * The add button was clicked
     *
     * @param e action event
     */
    private void addBtnClicked(ActionEvent e) {
        String className = (new InputStringDialog((Frame)null, this, "Enter class name for unit to be added:")).showDialog();
        if (!className.equals("")) {
            try {
                Class unitClass = Class.forName(className);
                if ((unitClass.getSuperclass()!=null) && (!unitClass.getSuperclass().getName().equals("model.AUnit"))) {
                    throw new RuntimeException("Invalid unit class, superclass is not model.AUnit<M extends IMeasurable>: "+unitClass.getSuperclass().getName());
                }
                Type ct = unitClass.getGenericSuperclass();
                if ((ct!=null) && (ct instanceof ParameterizedType)) {
                    ParameterizedType pt = (ParameterizedType)ct;
                    Type[] args = pt.getActualTypeArguments();
                    if ((args.length != 1) || (!(args[0] instanceof Class))) {
                        if (!unitClass.getSuperclass().getName().equals("model.AUnit")) {
                            throw new RuntimeException("Invalid unit class, superclass is not model.AUnit<M extends IMeasurable>: "+unitClass.getSuperclass().getName());
                        }
                    }
                    String measurableName = ((Class)args[0]).getName();
                    String selectedMeasurableName = _measurablesBox.getSelectedItem().toString();
                    if (!measurableName.equals(selectedMeasurableName)) {
                        throw new RuntimeException("Measurable of unit " + className + " is " + measurableName
                                                       + ", not " + selectedMeasurableName);
                    }
                    Constructor unitCtor = unitClass.getConstructor();
                    AUnit<? extends IMeasurable> unit = (AUnit<? extends IMeasurable>)unitCtor.newInstance();
                    ((IMeasurableChoice)_measurablesBox.getSelectedItem()).addUnit(unit);
                }
                else {
                    throw new RuntimeException("Invalid unit class, superclass is not model.AUnit<M extends IMeasurable>.");
                }
            }
            catch(Exception e1) {
                JOptionPane.showMessageDialog(this, e1);
            }
        }
    }
    
    /**
     * The convert button has been clicked.
     *
     * @param e action event
     */
    private void convertBtnClicked(ActionEvent e) {
        int count = _unitList.getSelectedValues().length;
        if (count > 1) {
            ArrayList<AUnit> selectedUnits = new ArrayList<AUnit>(count);
            for(Object o : _unitList.getSelectedValues()) {
                selectedUnits.add((AUnit<? extends IMeasurable>)o);
            }
            ConversionView cv = new ConversionView(selectedUnits);
            cv.setLocationRelativeTo(this);
            cv.setVisible(true);        }
        else {
            JOptionPane.showMessageDialog(this, "Please select two or more units.");
        }
    }
    
    /**
     * An item in the measurables box has been selected.
     *
     * @param e action event
     */
    private void measurablesBoxSelected(ActionEvent e) {
        ((IMeasurableChoice)_measurablesBox.getSelectedItem()).select();
    }
    
    /**
     * An entry in the measurable box.
     */
    private interface IMeasurableChoice {
        /**
         * Select this entry.
         */
        void select();
        
        /**
         * Add a unit to this measurable choice.
         *
         * @param unit unit to add
         */
        void addUnit(AUnit<? extends IMeasurable> unit);
    }
    
    /**
     * An entry in the measurable box that represents an actual measurable.
     */
    private class ConcreteMeasurableChoice implements IMeasurableChoice {
        /**
         * Name of this measurable.
         */
        String _name;
        
        /**
         * List of units belonging to this measurable.
         */
        DefaultListModel _unitModel;
        
        /**
         * Constructor for a concrete measurable choice.
         *
         * @param name  name of measurable
         * @param units units belonging to this measurable
         */
        public ConcreteMeasurableChoice(String name, AUnit<? extends IMeasurable>[] units) {
            _name = name;
            _unitModel = new DefaultListModel();
            for(AUnit<? extends IMeasurable> u : units) {
                _unitModel.addElement(u);
            }
        }
        
        /**
         * Returns a string representation of this measurable choice.
         *
         * @return string representation
         */
        public String toString() {
            return _name;
        }
        
        /**
         * Select this entry.
         */
        public void select() {
            _unitList.setModel(_unitModel);
            ConvCalcView.this.validate();
        }
        
        /**
         * Add a unit to this measurable choice.
         *
         * @param unit unit to add
         */
        public void addUnit(AUnit<? extends IMeasurable> unit) {
            _unitModel.addElement(unit);
            if (_measurablesBox.getSelectedItem() == this) {
                select();
            }
        }
    }
    
    /**
     * An entry in the measurable box that allows the user to add new measurable choices.
     */
    private class AddMeasurableChoice implements IMeasurableChoice {
        /**
         * Returns a string representation of this measurable choice.
         *
         * @return string representation
         */
        public String toString() {
            return "Add...";
        }
        
        /**
         * Select this entry.
         */
        public void select() {
            String className
                = (new InputStringDialog((Frame)null, ConvCalcView.this, "Enter class name for measurable to be added:")).showDialog();
            if (!className.equals("")) {
                try {
                    Class measurableClass = Class.forName(className);
                    if ((!measurableClass.isInterface()) || (!IMeasurable.class.isAssignableFrom(measurableClass))) {
                        throw new RuntimeException("Invalid measurable interface or superinterface is not model.IMeasurable.");
                    }
                    ConcreteMeasurableChoice newMeasurableChoice
                        = new ConcreteMeasurableChoice(className, new AUnit[0]);
                    _measurablesBox.insertItemAt(newMeasurableChoice, _measurablesBox.getItemCount() - 1);
                    _measurablesBox.setSelectedItem(newMeasurableChoice);
                    newMeasurableChoice.select();
                }
                catch(Exception e1) {
                    JOptionPane.showMessageDialog(ConvCalcView.this, e1);
                }
            }
        }
        
        /**
         * Add a unit to this measurable choice.
         * @param unit unit to add
         */
        public void addUnit(AUnit<? extends IMeasurable> unit) {
            throw new Error("Unit added to \"Add...\" measurable choice. This should never happen!");
        }
    }
}
