package view;

import model.IModelAdapter;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * Console view of the simple temperature calculator.
 *
 * @author Mathias Ricken
 */
public class TempCalcConsoleView {
    /**
     * Adapter to let the model perform computations.
     */
    private IModelAdapter _modelAdapter;
    
    /**
     * Buffered reader from standard in.
     */
    private BufferedReader _br = new BufferedReader(new InputStreamReader(System.in));
    
    /**
     * Constructs a new temperature calculator console view.
     *
     * @param modelAdapter adapter to model
     */
    public TempCalcConsoleView(IModelAdapter modelAdapter) {
        _modelAdapter = modelAdapter;
    }
    
    /**
     * Run the calculator.
     */
    public void run() {
        System.out.println("Convert temperature between Celsius and Fahrenheit.");
        String s;
        do {
            s = multipleChoice("Press 'c' to convert C to F, or 'f' to convert F to c.", "c", "f");
            if (s.equals("c")) {
                convertCtoF();
            }
            else {
                convertFtoC();
            }
            System.out.println("Do you want to convert more temperatures? ");
            s = multipleChoice("Press 'y' for yes, 'n' for no.", "y", "n");
        } while(s.equals("y"));
    }
    
    /**
     * Ask user to pick one of multiple choices.
     * @param prompt prompt to display
     * @param choices varargs with different choices
     * @return selected choice
     */
    private String multipleChoice(String prompt, String... choices) {
        String s = null;
        boolean repeat = true;
        do {
            System.out.print(prompt+" ");
            try {
                s = _br.readLine();
            }
            catch(IOException ioe) {
                continue;
            }
            repeat = true;
            for(String c: choices) {
                if (c.equals(s)) {
                    repeat = false;
                    break;
                }
            }
        } while(repeat);
        return s;
    }
    
    /**
     * Perform a temperature conversion from Celsius to Fahrenheit.
     */
    private void convertCtoF() {
        System.out.println();
        
        double d;
        boolean repeat = true;
        do {
            System.out.print("Please enter temperature in Celsius: ");
            try {
                d = Double.valueOf(_br.readLine());
                System.out.println(String.valueOf(d)+" C = "+String.valueOf(_modelAdapter.convertCtoF(d))+" F");
                repeat = false;
            }
            catch(NumberFormatException nfe) {
                // nothing to do, just repeat
            }
            catch(IOException ioe) {
                // nothing to do, just repeat
            }
        } while(repeat);
    }
    
    /**
     * Perform a temperature conversion from Fahrenheit to Celsius.
     */
    private void convertFtoC() {
        System.out.println();
        
        double d;
        boolean repeat = true;
        do {
            System.out.print("Please enter temperature in Fahrenheit: ");
            try {
                d = Double.valueOf(_br.readLine());
                System.out.println(String.valueOf(d)+" F = "+String.valueOf(_modelAdapter.convertFtoC(d))+" C");
                repeat = false;
            }
            catch(NumberFormatException nfe) {
                // nothing to do, just repeat
            }
            catch(IOException ioe) {
                // nothing to do, just repeat
            }
        } while(repeat);
    }
}
