package sysModel;

import controller.IEditAdapter;
import controller.IScrollAdapter;
import model.ILambda;
import sysModel.env.*;
import sysModel.fish.AFish;
import sysModel.fish.DynamicFishFactory;
import sysModel.parser.Parser;
import sysModel.parser.ParserException;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.FileWriter;
import java.io.PrintWriter;

/**
 * Simulation control coordinating interaction between view, simulation engine and environment.
 *
 * @author Mathias Ricken
 */
public class SimDriver {
    /**
     * Environment.
     */
    private AGlobalEnv _env;

    /**
     * Simulation engine.
     */
    private SimEngine _simEngine;

    /**
     * Command to set the seed when a new environment is loaded or created.
     */
    private ILambda _seedCommand;

    /**
     * Command executed before the simulation starts running.
     */
    private ILambda _startCommand;

    /**
     * Command executed after the simulation finishes an iteration.
     */
    private ILambda _iterationCommand;

    /**
     * Simulation speed.
     */
    private int _simulationSpeed;

    /**
     * Processing timer.
     */
    private Timer _timer;

    /**
     * Edit adapter.
     */
    private IEditAdapter _editAdapter = null;

    /**
     * Command factory for use in the environment.
     */
    private ICmdFactory _cmdFactory = new ICmdFactory() {
        public ILambda makeNotifyCmd(final ILambda lambda) {
            return new ILambda() {
                public Object apply(Object param) {
                    _simEngine.notify(lambda);
                    return null;
                }
            };
        }

        public ILambda makeDeleteCmd(final ILocalEnv env) {
            return makeNotifyCmd(new ILambda() {
                public Object apply(Object param) {
                    FishApplyParams applyParams = (FishApplyParams) param;
                    // if the fish is at this location
                    if (applyParams.localEnv() == env) {
                        // ask the simulation engine to remove it
                        _simEngine.delete(applyParams.fish());
                    }
                    return null;
                }
            });
        }

        public ILambda makeAddCmd(final AFish fish) {
            return new ILambda() {
                public Object apply(Object param) {
                    _simEngine.add(fish);
                    return null;
                }
            };
        }
    };

    /**
     * Construct a new model.
     */
    public SimDriver() {
        _env = NullEnv.instance();
        _simEngine = new SimEngine();
        _seedCommand = NoOpLambda.instance();
        _startCommand = NoOpLambda.instance();
        _iterationCommand = NoOpLambda.instance();
    }

    /**
     * Get the simulation engine.
     *
     * @return simulation engine
     */
    public ICmdFactory getCmdFactory() {
        return _cmdFactory;
    }

    /**
     * Set adapters.
     *
     * @param ea makeEditCmd adapter to use
     */
    public void setAdapters(IEditAdapter ea) {
        _editAdapter = ea;
    }

    /**
     * Load environment from file.
     *
     * @param filename filename
     * @return true if successful
     */
    public boolean loadEnvironment(String filename) {
        try {
            _simEngine.clear();
            _env = Parser.parse(filename, _cmdFactory);
            _seedCommand.apply(null);
            return true;
        }
        catch (ParserException e) {
            System.out.println(e);
            return false;
        }
    }

    /**
     * Save environment to file.
     *
     * @param filename filename
     * @return true if successful
     */
    public boolean saveEnvironment(String filename) {
        try {
            // open a new PrintWriter
            PrintWriter pw = new PrintWriter(new FileWriter(filename));

            // ask the environment what to do
            ILambda lambda = _env.save(pw);

            // and do it
            _simEngine.notify(lambda);

            // close the PrintWriter
            pw.close();
            return true;
        }
        catch (java.io.IOException e) {
            return false;
        }
    }

    /**
     * Create environment from factory.
     *
     * @param factory factory
     * @return true if successful
     */
    public boolean createEnvironment(AEnvFactory factory) {
        _env = factory.create();
        _simEngine.clear();
        _seedCommand.apply(null);
        return true;
    }

    /**
     * Set the seed command. This command gets executed before a new
     * environment is created or loaded.
     *
     * @param seedCmd command to set
     */
    public void setSeedLambda(ILambda seedCmd) {
        _seedCommand = seedCmd;
    }

    /**
     * Make a simulation step.
     */
    public void step() {
        // ask the environment what to do
        ILambda lambda = _env.makeStepCmd();

        // and do it
        lambda.apply(this);
    }

    /**
     * Start simulation.
     */
    public void start() {
        _startCommand.apply(null);
        _timer = new Timer(_simulationSpeed, new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                step();
                _iterationCommand.apply(null);
            }
        });
        _timer.start();
    }

    /**
     * Stop simulation.
     */
    public void stop() {
        if (_timer != null) {
            _timer.stop();
        }
    }

    /**
     * Change the simulation speed.
     *
     * @param speed simulation speed
     */
    public void setSpeed(int speed) {
        _simulationSpeed = speed;
        if (_timer != null) {
            _timer.setDelay(speed);
        }
    }

    /**
     * Set simulation start command. This command gets called
     * before the simulation starts.
     *
     * @param startCmd start command
     */
    public void setStartLambda(ILambda startCmd) {
        _startCommand = startCmd;
    }

    /**
     * Set simulation iteration command. This command gets called
     * after the simulation finishes a step.
     *
     * @param itCmd start command
     */
    public void setIterationLambda(ILambda itCmd) {
        _iterationCommand = itCmd;
    }

    /**
     * Draw model in this region. The graphics object has been set up so that
     * (0,0) represents the top left and (100,100) the bottom right corner.
     *
     * @param g    graphics object
     * @param comp component to drawFish on
     * @param p1   top left corner of the region
     * @param p2   bottom right corner of the region
     */
    public void draw(Graphics2D g, Component comp, Point.Double p1, Point.Double p2) {
        // ask the environment what to do
        ILambda lambda = _env.makeDrawCmd(g, comp, p1, p2);

        // and do it
        lambda.apply(this);
    }

    /**
     * Edit a field.
     *      @param p coordinates in model coordinate units
     @param button
     */
    public void edit(Point.Double p, int button) {
        // create the fishFactory to use
        DynamicFishFactory fishFactory = new DynamicFishFactory(_editAdapter.getCurrentFish(), _editAdapter.getCurrentColor());

        // let the environment decide what to do
        ILambda lambda = _env.makeEditCmd(p, fishFactory, button);

        // and do it
        lambda.apply(this);
    }

    /**
     * The action to be executed if the display should return home.
     *
     * @param sa scroll adapter
     */
    public void returnHome(final IScrollAdapter sa) {
        _env.returnHome(sa);
    }

    /**
     * Get size of the display.
     *
     * @return size of the display in model coordinate units.
     */
    public Dimension getDisplaySize() {
        return _env.getDisplaySize();
    }

    /**
     * Ask the model where to scroll, given where the user has scrolled.
     * If the environment just acts like a normal panal, it should return pos without modification.
     * If the environment recenters, it should return a position in the middle of the pan area.
     * All coordinates are in model coordinate units.
     *
     * @param pos position where the user scrolled to
     * @return position where the environment wants the view to be
     * @see controller.IDisplayAdapter#getPanDelta
     */
    public Point.Double getViewPosition(Point.Double pos) {
        return _env.getViewPosition(pos);
    }

    /**
     * Ask the model how much to pan, given where the user scrolled.
     * If the environment just acts like a normal panal, it should return (0,0).
     * If the environment recenters, it should return delta without modification.
     * All coordinates are in model coordinate units.
     *
     * @param delta how far the user scrolled
     * @return how far the panel should scroll
     * @see controller.IDisplayAdapter#getViewPosition
     */
    public Point.Double getPanDelta(Point.Double delta) {
        return _env.getPanDelta(delta);
    }

    /**
     * Get a tool tip description for a specific place in the environment.
     *
     * @param p mouse coordinates
     * @return tool tip text
     */
    public String getToolTipText(Point.Double p) {
        return ((String) _env.getToolTipText(p).apply(this));
    }
}
