package sysModel.env;

import controller.IScrollAdapter;
import junit.framework.TestCase;
import model.ILambda;
import model.fish.GenericFish;
import sysModel.ICmdFactory;
import sysModel.NoOpLambda;
import sysModel.fish.AFish;
import sysModel.fish.IFishFactory;
import sysModel.parser.DefaultTokenVisitor;
import sysModel.parser.Lexer;
import sysModel.parser.ParserException;

import javax.swing.*;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.util.LinkedList;

/**
 * Implementation of a square bounded environment.
 *
 * @author Mathias G. Ricken
 */
public class BoundedEnv extends ASquareEnv {
    /**
     * Field interface.
     */
    public static interface IField {
        /**
         * Visitor hook.
         *
         * @param vis   visitor to execute.
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public abstract Object execute(IFieldVisitor vis, Object param);
    }

    /**
     * Empty field singleton.
     */
    public static class EmptyField implements IField {
        /**
         * Singleton instance.
         */
        public static final EmptyField Singleton = new EmptyField();

        /**
         * Singleton ctor.
         */
        private EmptyField() {
        }

        /// call empty case
        public Object execute(IFieldVisitor vis, Object param) {
            return vis.emptyCase(this, param);
        }
    }

    /**
     * Non-empty field class.
     */
    public static class NonEmptyField implements IField {
        /// local environment in this field
        private LocalEnvironment _localEnv;

        /**
         * Constructor.
         *
         * @param localEnv local environment in this field
         */
        public NonEmptyField(LocalEnvironment localEnv) {
            _localEnv = localEnv;
        }

        /**
         * Return local environment.
         *
         * @return local environment in this field
         */
        public LocalEnvironment getLocalEnv() {
            return _localEnv;
        }

        /**
         * Set local environment.
         *
         * @param localEnv environment in this field
         */
        public void setLocalEnv(LocalEnvironment localEnv) {
            _localEnv = localEnv;
        }

        /// call non-empty case
        public Object execute(IFieldVisitor vis, Object param) {
            return vis.nonEmptyCase(this, param);
        }
    }

    /**
     * Field visitor interface.
     */
    public static interface IFieldVisitor {
        /**
         * Empty case.
         *
         * @param host  empty field
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public abstract Object emptyCase(EmptyField host, Object param);

        /**
         * Non-empty case.
         *
         * @param host  non-empty field
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public abstract Object nonEmptyCase(NonEmptyField host, Object param);
    }

    /**
     * Concrete local environment for the square bounded environment.
     */
    protected class LocalEnvironment implements ISquareLocalEnvironment {
        /**
         * Location.
         */
        Location _loc;

        /**
         * Direction.
         */
        Direction _dir;

        /**
         * State.
         */
        ILocalEnvState _state = EmptyLocalEnvState.Singleton;

        /**
         * Lambda to execute a move.
         */
        private class MoveLambda implements ILambda {
            /// target direction
            private Direction _newDir;
            /// target location
            private Location _newLoc;

            /**
             * Constructor.
             *
             * @param le target local environment
             */
            public MoveLambda(LocalEnvironment le) {
                _newLoc = makeLocation(le._loc.getX(), le._loc.getY());
                _newDir = makeDirection(le._dir);
            }

            /**
             * Execute the move.
             *
             * @param param not used
             * @return null
             */
            public Object apply(Object param) {
                _fieldMap[(int) _loc.getX()][(int) _loc.getY()] = EmptyField.Singleton;

                // execute the movement
                _loc = _newLoc;
                _dir = _newDir;

                _fieldMap[(int) _loc.getX()][(int) _loc.getY()] = new NonEmptyField(LocalEnvironment.this);

                // deactivate all lambdas
                deactivateMoveLambdas();
                return null;
            }
        }

        /**
         * Construct a new local environment.
         *
         * @param loc location
         * @param dir direction
         */
        public LocalEnvironment(Location loc, Direction dir) {
            _loc = loc;
            _dir = dir;
        }

        /**
         * Accessor for the location.
         *
         * @return location
         */
        public Location location() {
            return _loc;
        }

        /**
         * Accessor for the direction.
         *
         * @return direction
         */
        public Direction direction() {
            return _dir;
        }

        /**
         * Make local environment in forward direction. Do not block yourself.
         *
         * @return new local environment in forward direction
         */
        private ILocalEnv makeMoveFwdLocalEnv() {
            // remove this local environment to prevent collision with itself
            _fieldMap[(int) location().getX()][(int) location().getY()] = EmptyField.Singleton;
            ILocalEnv le = makeLocalEnv(_loc.getNeighbor(_dir), _dir);
            // add this local environment back in
            _fieldMap[(int) location().getX()][(int) location().getY()] = new NonEmptyField(LocalEnvironment.this);
            return le;
        }

        /**
         * Attempt to move the fish forward, which may or may not be successful.
         * The behavior in each case is defined by the visitor:
         * - If the move cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
         * - If the move can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
         * executed to actually move the fish to the target location of this move. The ILambda ignores the input
         * parameter and returns null.
         *
         * @param fish       AFish to move
         * @param blockedCmd lambda to apply if blocked
         * @param openCmd    lambda to apply if open
         * @return return value of lambda executed
         */
        public Object tryMoveFwd(AFish fish, final ILambda blockedCmd, final ILambda openCmd) {
            // TODO: PART3
            return null;
        }

        /**
         * Attempt to breed the fish forward, which may or may not be successful.
         * The behavior in each case is defined by the visitor:
         * - If the breeding cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
         * - If the breeding can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
         * executed to actually breed the fish to the target location of this breeding. The ILambda ignores the input
         * parameter and returns null.
         *
         * @param fish       AFish to move
         * @param blockedCmd lambda to apply if blocked
         * @param openCmd    lambda to apply if open
         * @return return value of lambda executed
         */
        public Object tryBreedFwd(final AFish fish, final ILambda blockedCmd, final ILambda openCmd) {
            // TODO: PART3
            return null;
        }

        /**
         * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
         *
         * @param fish AFish to drawFish
         * @param g    graphics object to drawFish on
         * @param comp component to drawFish on
         */
        public void drawFish(AFish fish, Graphics2D g, Component comp) {
            final double centerX = Math.floor(_loc.getX()) + 1.0 / 2;
            final double centerY = Math.floor(_loc.getY()) + 1.0 / 2;

            // save transformation
            AffineTransform oldTransform = g.getTransform();
            // translate to center of field
            g.translate(centerX, centerY);

            // set up the correct rotation
            _dir.rotateGraphics(g);

            // makeDrawCmd the fish
            fish.paint(g, comp);

            // restore transformation
            g.setTransform(oldTransform);
        }

        /**
         * Turn the fish radians to the right.
         *
         * @param fish    AFish to turn
         * @param radians radians to turn
         */
        public void turnRight(AFish fish, double radians) {
            _dir.turnRight(radians);
        }

        /**
         * Remove the fish from the environment.
         *
         * @param fish AFish to remove
         */
        public void removeFish(AFish fish) {
            BoundedEnv.this.removeFish(this);
        }

        /**
         * Execute a visitor on this local environment.
         *
         * @param visitor visitor to execute
         * @param param   visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object execute(final ILocalEnvVisitor visitor, final Object param) {
            return _state.execute(this, visitor, param);
        }

        /**
         * String representation of the local environment.
         * Should be "(x, y) (dx, dy)".
         *
         * @return string representation
         */
        public String toString() {
            return _loc.toString() + " " + _dir.toString();
         }

        /**
         * Change the state of this local environment.
         *
         * @param state new state
         */
        public void setState(ILocalEnvState state) {
            _state = state;
        }

        /**
         * Return true of location is outside of environment.
         *
         * @param l location
         * @return true if outside
         */
        public boolean outOfRange(Location l) {
            long x = (int) Math.floor(l.getX());
            long y = (int) Math.floor(l.getY());
            return x < 0 || y < 0 || x >= _width || y >= _height;
        }
    }

    /**
     * Size of the pan area.
     */
    protected final int PAN_SIZE = 2000;

    /**
     * Center of the pan area.
     */
    protected final Point.Double PAN_CENTER = new Point.Double(PAN_SIZE / 2, PAN_SIZE / 2);

    /**
     * List of local environments in this global environment.
     */
    protected IField[][] _fieldMap;

    /**
     * Width.
     */
    protected int _width;

    /**
     * Height.
     */
    protected int _height;

    /**
     * Construct a new square bounded environment.
     * Does not set this object up for actual use.
     * Note: This constructor needs to exist and be public for the "environment selection"
     * dialog to work.
     *
     * @param cmdFactory command factory to use
     */
    public BoundedEnv(ICmdFactory cmdFactory) {
        super(cmdFactory);
        _fieldMap = new IField[0][0];
    }

    /**
     * Construct a new square bounded environment.
     *
     * @param cmdFactory command factory to use
     * @param width      width of environment
     * @param height     height of environment
     */
    public BoundedEnv(ICmdFactory cmdFactory, int width, int height) {
        super(cmdFactory);
        _width = width;
        _height = height;
        _fieldMap = new IField[width][height];
        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                _fieldMap[x][y] = EmptyField.Singleton;
            }
        }
    }

    /**
     * Add the fish to the global environment.
     *
     * @param localEnv local environment
     * @param fish     fish to add
     */
    protected void addFishToInternalData(ILocalEnv localEnv, AFish fish) {
        _fieldMap
                [(int) ((LocalEnvironment) localEnv).location().getX()]
                [(int) ((LocalEnvironment) localEnv).location().getY()] = new NonEmptyField((LocalEnvironment) localEnv);
    }

    /**
     * Remove the fish from the global environment.
     *
     * @param localEnv local environment
     */
    protected void removeFishFromInternalData(ILocalEnv localEnv) {
        _fieldMap
                [(int) ((LocalEnvironment) localEnv).location().getX()]
                [(int) ((LocalEnvironment) localEnv).location().getY()] = EmptyField.Singleton;
    }

    /**
     * Create a local environment for the position.
     *
     * @param loc location
     * @param dir direction
     * @return local environment
     */
    protected ISquareLocalEnvironment makeLocalEnv(final Location loc, final Direction dir) {
        return ((ISquareLocalEnvironment) _fieldMap[(int) loc.getX()][(int) loc.getY()].execute(new IFieldVisitor() {
            public Object emptyCase(EmptyField host, Object inp) {
                return new LocalEnvironment(loc, dir);
            }

            public Object nonEmptyCase(NonEmptyField host, Object inp) {
                return host.getLocalEnv();
            }
        }, null));
    }

    /**
     * Create a local environment with the given data.
     *
     * @param loc location
     * @param dir direction
     * @return new local environment
     */
    protected ISquareLocalEnvironment createLocalEnvironment(Location loc, Direction dir) {
        return new LocalEnvironment(loc, dir);
    }

    /**
     * Factory method for parsing a stream of tokens and creating a global environment from it.
     *
     * @param l lexer
     * @return new global environment
     */
    protected AGlobalEnv parseEnvironment(final Lexer l) {
        // have to read size of environment
        return (AGlobalEnv) l.nextToken().execute(new DefaultTokenVisitor() {
            public Object defaultCase() {
                throw new ParserException("Invalid token");
            }

            public Object numCase(final double width) {
                // width was read
                return (AGlobalEnv) l.nextToken().execute(new DefaultTokenVisitor() {
                    public Object defaultCase() {
                        throw new ParserException("Invalid token");
                    }

                    public Object numCase(final double height) {
                        // height was read
                        BoundedEnv env = new BoundedEnv(_cmdFactory, (int) width, (int) height);

                        // parse fish
                        env.parseFish(l);

                        return env;
                    }
                });
            }
        });
    }

    /**
     * Get the environment settings class.
     *
     * @return environment settings class
     */
    public AEnvFactory makeEnvFactory() {
        return new AEnvFactory() {
            private JTextField _rowField;
            private JTextField _colField;

            {
                setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
                add(new JLabel("rows: "));
                add(_rowField = new JTextField("10"));
                add(new JLabel("  cols: "));
                add(_colField = new JTextField("10"));
            }

            public AGlobalEnv create() {
                return new BoundedEnv(BoundedEnv.this._cmdFactory, Integer.parseInt(_colField.getText()), Integer.parseInt(_rowField.getText()));
            };
            public String toString() {
                return BoundedEnv.class.getName();
            }
        };
    }

    /**
     * Print file header.
     *
     * @param pw PrintWriter to use
     */
    protected void printHeader(java.io.PrintWriter pw) {
        pw.println(this.getClass().getName());
    }

    /**
     * Get size of the display.
     *
     * @return size of the display in model coordinate units.
     */
    public Dimension getDisplaySize() {
        return new Dimension(_width, _height);
    }

    /**
     * The action to be executed if the display should return home.
     *
     * @param sa scroll adapter
     */
    public void returnHome(final IScrollAdapter sa) {
        sa.setCorner(0, 0);
    }

    /**
     * Ask the model where to scroll, given where the user has scrolled.
     * If the environment just acts like a normal panal, it should return pos without modification.
     * If the environment recenters, it should return a position in the middle of the pan area.
     * All coordinates are in model coordinate units.
     *
     * @param pos position where the user scrolled to
     * @return position where the environment wants the view to be
     * @see controller.IDisplayAdapter#getPanDelta
     */
    public Point.Double getViewPosition(Point.Double pos) {
        // the panel just acts like a normal panal, return position without any changes
        return pos;
    }

    /**
     * Ask the model how much to pan, given where the user scrolled.
     * If the environment just acts like a normal panal, it should return (0,0).
     * If the environment recenters, it should return delta without modification.
     * All coordinates are in model coordinate units.
     *
     * @param delta how far the user scrolled
     * @return how far the panel should scroll
     * @see controller.IDisplayAdapter#getViewPosition
     */
    public Point.Double getPanDelta(Point.Double delta) {
        // we do not want the panel to keep track of position, do not pan
        return new Point.Double(0, 0);
    }



    /*****************************************************************************************************************
     * Tests follow
     *****************************************************************************************************************/

    /**
     * Test cases for BoundedEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_BoundedEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private BoundedEnv _env;
        private IFishFactory _fishFactory;

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new BoundedEnv(_cmdFactory, 10, 10);

            _fishFactory = new IFishFactory() {
                /**
                 * Create a new fish.
                 *
                 * @return new fish
                 */
                public AFish createFish() {
                    return new GenericFish(Color.RED);
                }
            };
        }

        /**
         * Test outOfRange.
         */
        public void testOutOfRange() {
            LocalEnvironment l = (LocalEnvironment)_env.makeLocalEnv(new Point2D.Double());
            assertEquals(false, l.outOfRange(_env.makeLocation(0, 0)));
            assertEquals(false, l.outOfRange(_env.makeLocation(9, 9)));
            assertEquals(true, l.outOfRange(_env.makeLocation(10, 0)));
            assertEquals(true, l.outOfRange(_env.makeLocation(0, 10)));
            assertEquals(true, l.outOfRange(_env.makeLocation(10, 10)));
            assertEquals(true, l.outOfRange(_env.makeLocation(-1, 0)));
            assertEquals(true, l.outOfRange(_env.makeLocation(0, -1)));
            assertEquals(true, l.outOfRange(_env.makeLocation(-1, -1)));
            assertEquals(true, l.outOfRange(_env.makeLocation(-1, 9)));
            assertEquals(true, l.outOfRange(_env.makeLocation(9, -1)));
        }

        int countNonEmpty() {
            int count = 0;
            BoundedEnv.IField[][] map = _env._fieldMap;
            for (int i = 0; i < map.length; i++) {
                BoundedEnv.IField[] iFields = map[i];
                for (int j = 0; j < iFields.length; j++) {
                    BoundedEnv.IField iField = iFields[j];
                    count += ((Integer) iField.execute(new BoundedEnv.IFieldVisitor() {
                        public Object emptyCase(BoundedEnv.EmptyField host, Object param) {
                            return new Integer(0);
                        }

                        public Object nonEmptyCase(BoundedEnv.NonEmptyField host, Object param) {
                            return new Integer(1);
                        }
                    }, null)).intValue();
                }
            }
            return count;
        }

        /**
         * Test addFishToInternalData.
         */
        public void testAddFish() {
            assertEquals(0, countNonEmpty());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFishToInternalData(lTop, fTop);

            assertEquals(1, countNonEmpty());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFishToInternalData(lBottom, fBottom);

            assertEquals(2, countNonEmpty());
        }

        /**
         * Test editFish.
         */
        public void testEditFish() {
            assertEquals(0, countNonEmpty());

            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFishToInternalData(l, f);
            BoundedEnv.Direction d = l.direction();

            assertEquals(1, countNonEmpty());
            assertEquals(0.0, d.getAngle(), 0.01);

            ILambda lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, countNonEmpty());
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, countNonEmpty());
            assertEquals(Math.PI, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, countNonEmpty());
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(0, countNonEmpty());
            assertEquals(0, d.getAngle(), 0.01);
            assertEquals(_delete, lambda);
        }

        /**
         * Test getViewPosition.
         */
        public void testGetViewPosition() {
            assertTrue(_env.getViewPosition(new Point.Double(0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(1.0, 0)).equals(new Point.Double(1.0, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(1.2, 0)).equals(new Point.Double(1.2, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.0)).equals(new Point.Double(0, 1.0)));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.3)).equals(new Point.Double(0, 1.3)));
            assertTrue(_env.getViewPosition(new Point.Double(-2.5, 0)).equals(new Point.Double(-2.5, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(-3.0, 0)).equals(new Point.Double(-3.0, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(0, -2.5)).equals(new Point.Double(0, -2.5)));
            assertTrue(_env.getViewPosition(new Point.Double(0, -3.0)).equals(new Point.Double(0, -3.0)));
            assertTrue(_env.getViewPosition(new Point.Double(2.0, 1.0)).equals(new Point.Double(2.0, 1.0)));
            assertTrue(_env.getViewPosition(new Point.Double(-4.0, -2.3)).equals(new Point.Double(-4.0, -2.3)));
        }

        /**
         * Test getPanDelta.
         */
        public void testGetPanDelta() {
            assertTrue(_env.getPanDelta(new Point.Double(0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.2, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.3)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-2.5, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-3.0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -2.5)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -3.0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(2.0, 1.0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-4.0, -2.3)).equals(new Point.Double(0, 0)));
        }
    }

    /**
     * Test cases for BoundedEnv.LocalEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_BoundedEnv_LocalEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private BoundedEnv _env;

        private static class SuccessException extends RuntimeException {
            public SuccessException() {
                super();
            }
        }

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new BoundedEnv(_cmdFactory, 10, 10);
        }

        /**
         * Test local environment's execute.
         */
        public void testExecute() {
            ILocalEnv l = _env.makeLocalEnv(new Point.Double(1, 1));

            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch(SuccessException e) { }


            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            l.setState(NonEmptyLocalEnvState.Singleton);
            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }

            ILocalEnv l2 = _env.makeLocalEnv(new Point.Double(1, 1));
            try {
                l2.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }

            ILocalEnv l3 = _env.makeLocalEnv(new Point.Double(1.4, 1.6));
            try {
                l3.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }

            ILocalEnv l4 = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch(SuccessException e) { }

            GenericFish f4 = new GenericFish(Color.RED);
            f4.setLocalEnvironment(l4);
            _env.addFish(l4, f4);
            l4.setState(NonEmptyLocalEnvState.Singleton);
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }
        }

        /**
         * Test local environment's tryMoveFwd.
         */
        public void testTryMoveFwd() {
            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);
            lTop.setState(NonEmptyLocalEnvState.Singleton);

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);
            lBottom.setState(NonEmptyLocalEnvState.Singleton);

            // move lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // move lTop --> open, don't move
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            // move lBottom into lTop --> blocked
            i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(789), i);

            // move lTop --> open, move
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, make move
                    ((ILambda) param).apply(null);
                    return new Integer(111);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(111), i);

            // move lBottom --> open
            i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(222);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(222), i);

            // move lTop into edge --> blocked
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(333);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(333), i);

            // turn and move lTop --> open
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(444);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(444), i);
        }

        int countNonEmpty() {
            int count = 0;
            BoundedEnv.IField[][] map = _env._fieldMap;
            for (int i = 0; i < map.length; i++) {
                BoundedEnv.IField[] iFields = map[i];
                for (int j = 0; j < iFields.length; j++) {
                    BoundedEnv.IField iField = iFields[j];
                    count += ((Integer) iField.execute(new BoundedEnv.IFieldVisitor() {
                        public Object emptyCase(BoundedEnv.EmptyField host, Object param) {
                            return new Integer(0);
                        }

                        public Object nonEmptyCase(BoundedEnv.NonEmptyField host, Object param) {
                            return new Integer(1);
                        }
                    }, null)).intValue();
                }
            }
            return count;
        }

        /**
         * Test local environment's tryBreedFwd.
         */
        public void testTryBreedFwd() {
            assertEquals(0, countNonEmpty());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);
            lTop.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(1, countNonEmpty());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);
            lBottom.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(2, countNonEmpty());

            // breed lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> open, don't breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            assertEquals(2, countNonEmpty());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            assertEquals(3, countNonEmpty());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // turn and breed lTop --> open, don't breed
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(3, countNonEmpty());

            // turn and breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(4, countNonEmpty());

            // turn and breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(4, countNonEmpty());
        }

        /**
         * Test local environment's turnRight.
         */
        public void testTurnRight() {
            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            l.setState(NonEmptyLocalEnvState.Singleton);
            BoundedEnv.Direction d = l.direction();

            assertEquals(0.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(0, d.getAngle(), 0.01);
        }

        /**
         * Test local environment's removeFish.
         */
        public void testRemoveFish() {
            assertEquals(0, countNonEmpty());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);
            lTop.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(1, countNonEmpty());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);
            lBottom.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(2, countNonEmpty());

            lTop.removeFish(fTop);

            assertEquals(1, countNonEmpty());

            lBottom.removeFish(fBottom);

            assertEquals(0, countNonEmpty());
        }

        /**
         * Test to make sure only one move lambda can be executed.
         */
        public void testOnlyOneMove() {
            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(5.0, 5.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            l.setState(NonEmptyLocalEnvState.Singleton);

            final LinkedList lambdas = new LinkedList();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            ((ILambda) lambdas.get(0)).apply(null);
            BoundedEnv.Location loc = l.location();
            assertTrue(loc.same(_env.makeLocation(5.0, 4.0)));

            ((ILambda) lambdas.get(1)).apply(null);
            loc = l.location();
            assertTrue(loc.same(_env.makeLocation(5.0, 4.0)));

            lambdas.clear();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            ((ILambda) lambdas.get(1)).apply(null);
            loc = l.location();
            assertTrue(loc.same(_env.makeLocation(6.0, 4.0)));

            ((ILambda) lambdas.get(0)).apply(null);
            loc = l.location();
            assertTrue(loc.same(_env.makeLocation(6.0, 4.0)));
        }
    }

}