package sysModel.env;

import model.ILambda;
import sysModel.fish.AFish;

import java.awt.*;

/**
 * Interface of the environment local to a fish.
 *
 * @author Mathias Ricken
 */
public interface ILocalEnv {
    /**
     * Attempt to move the fish forward, which may or may not be successful.
     * The behavior in each case is defined by the visitor:
     * - If the move cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
     * - If the move can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
     * executed to actually move the fish to the target location of this move. The ILambda ignores the input
     * parameter and returns null.
     *
     * @param fish       AFish to move
     * @param blockedCmd lambda to apply if blocked
     * @param openCmd    lambda to apply if open
     * @return return value of lambda executed
     */
    public abstract Object tryMoveFwd(AFish fish, ILambda blockedCmd, ILambda openCmd);

    /**
     * Attempt to breed the fish forward, which may or may not be successful.
     * The behavior in each case is defined by the visitor:
     * - If the breeding cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
     * - If the breeding can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
     * executed to actually move the fish to the target location of this breeding. The ILambda ignores the input
     * parameter and returns null.
     *
     * @param fish       AFish to move
     * @param blockedCmd lambda to apply if blocked
     * @param openCmd    lambda to apply if open
     * @return return value of lambda executed
     */
    public abstract Object tryBreedFwd(AFish fish, ILambda blockedCmd, ILambda openCmd);

    /**
     * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
     *
     * @param fish AFish to drawFish
     * @param g    graphics object to drawFish on
     * @param comp component to drawFish on
     */
    public abstract void drawFish(AFish fish, Graphics2D g, Component comp);

    /**
     * Turn the fish radians to the right.
     *
     * @param fish    AFish to turn
     * @param radians radians to turn
     */
    public abstract void turnRight(AFish fish, double radians);

    /**
     * Remove the fish from the environment.
     *
     * @param fish AFish to remove
     */
    public abstract void removeFish(AFish fish);

    /**
     * Execute a visitor on this local environment.
     *
     * @param visitor visitor to execute
     * @param param   visitor-specific parameter
     * @return visitor-specific return value
     */
    public abstract Object execute(AGlobalEnv.ILocalEnvVisitor visitor, Object param);

    /**
     * String representation of the local environment.
     * Should be "(x, y) (dx, dy)".
     *
     * @return string representation
     */
    public abstract String toString();

    /**
     * Set state.
     * @param state new state
     */
    public abstract void setState(ILocalEnvState state);
}
