package sysModel.fish;

import model.ILambda;
import model.RandNumGenerator;
import sysModel.env.FishApplyParams;
import sysModel.env.ILocalEnv;

import java.awt.*;
import java.util.Observable;
import java.util.Observer;

/**
 * Abstract class for a fish.
 * *
 *
 * @author Mathias G. Ricken
 */
public abstract class AFish implements Cloneable, Observer {
    /**
     * The local environment of the fish.
     */
    private ILocalEnv _localEnv;

    /**
     * The probability that this fish attempts to breed this timestep.
     * Subclasses can override this value.
     */
    protected double _probOfBreeding = 0; // 1.0f/7.0f;

    /**
     * The probability that this fish will die in this timestep.
     * Subclasses can override this value.
     */
    protected double _probOfDying = 0; // 1.0f/5.0f;

    /**
     * Color of the fish.
     */
    protected Color _fishColor;

    /**
     * Display for the fish.
     */
    protected IFishDisplay _fishDisplay;

    /**
     * Create a new fish.
     *
     * @param fishColor   fish color
     * @param fishDisplay fish display
     */
    public AFish(Color fishColor, IFishDisplay fishDisplay) {
        _fishColor = fishColor;
        _fishDisplay = fishDisplay;
    }

    /**
     * Set the local environment of this fish.
     *
     * @param localEnv local environment
     */
    public final void setLocalEnvironment(ILocalEnv localEnv) {
        _localEnv = localEnv;
    }

    /**
     * Attempt to move the fish forward, which may or may not be successful.
     * The behavior in each case is defined by the visitor:
     * - If the move cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
     * - If the move can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
     * executed to actually move the fish to the target location of this move. The ILambda ignores the input
     * parameter and returns null.
     *
     * @param blockedCmd lambda to apply if blocked
     * @param openCmd    lambda to apply if open
     * @return return value of lambda executed
     */
    public final Object tryMoveFwd(final ILambda blockedCmd, final ILambda openCmd) {
        // TODO: PART3
        return null;
    }

    /**
     * Attempt to breed the fish forward, which may or may not be successful.
     * The behavior in each case is defined by the visitor:
     * - If the breeding cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
     * - If the breeding can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
     * executed to actually move the fish to the target location of this breeding. The ILambda ignores the input
     * parameter and returns null.
     *
     * @param blockedCmd lambda to apply if blocked
     * @param openCmd    lambda to apply if open
     * @return return value of lambda executed
     */
    public final Object tryBreedFwd(final ILambda blockedCmd, final ILambda openCmd) {
        // TODO: PART3
        return null;
    }

    /**
     * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
     *
     * @param g    graphics object to drawFish on
     * @param comp component to drawFish on
     */
    public final void draw(Graphics2D g, Component comp) {
        _localEnv.drawFish(this, g, comp);
    }

    /**
     * Turn the fish radians to the right.
     *
     * @param radians radians to turn
     */
    public final void turnRight(double radians) {
        _localEnv.turnRight(this, radians);
    }

    /**
     * Turn the fish radians to the left.
     *
     * @param radians radians to turn
     */
    public final void turnLeft(double radians) {
        _localEnv.turnRight(this, -radians);
    }

    /**
     * Turn the fish Pi/2 radians to the right.
     */
    public void turnRight() {
        turnRight(Math.PI / 2);
    }

    /**
     * Turn the fish Pi/2 radians to the left.
     */
    public void turnLeft() {
        turnLeft(Math.PI / 2);
    }

    /**
     * Execute a simulation step. This function acts as a template method.
     * It always calls the breed, move and die functions, which can be
     * overridden, though.
     */
    public final void act() {
        breed();
        move();
        die();
    }

    /**
     * Execute the breeding part of a simulation step.
     */
    protected void breed() {
        if (RandNumGenerator.instance().nextDouble() < _probOfBreeding) {
            // breed
            // by default, a fish breeds into all open spaces around it
            final ILambda blocked1 = new ILambda() {
                public Object apply(Object param) {
                    // turn PI/2 radians to the right to face in the original direction
                    turnRight();
                    return null;
                }
            };

            final ILambda blocked2 = new ILambda() {
                public Object apply(Object param) {
                    // turn PI/2 radians to the right and attempt to breed forward
                    turnRight();
                    return tryBreedFwd(blocked1, new ILambda() {
                        public Object apply(Object param) {
                            // the field to the right is open
                            ((ILambda) param).apply(null);
                            return blocked1.apply(null);
                        }
                    });
                }
            };

            final ILambda blocked3 = new ILambda() {
                public Object apply(Object param) {
                    // turn PI/2 radians to the right and attempt to breed forward
                    turnRight();
                    return tryBreedFwd(blocked2, new ILambda() {
                        public Object apply(Object param) {
                            // the field to the right is open
                            ((ILambda) param).apply(null);
                            return blocked2.apply(null);
                        }
                    });
                }
            };

            final ILambda blocked4 = new ILambda() {
                public Object apply(Object param) {
                    // turn PI/2 radians to the right and attempt to breed forward
                    turnRight();
                    return tryBreedFwd(blocked3, new ILambda() {
                        public Object apply(Object param) {
                            // the field straight ahead is open
                            ((ILambda) param).apply(null);
                            return blocked3.apply(null);
                        }
                    });
                }
            };

            // attempt to breed forward
            tryBreedFwd(blocked4, new ILambda() {
                public Object apply(Object param) {
                    // the field to the left is open
                    ((ILambda) param).apply(null);
                    return blocked4.apply(null);
                }
            });
        }
    }

    /**
     * Execute the dying part of a simulation step.
     */
    protected void die() {
        if (RandNumGenerator.instance().nextDouble() < _probOfDying) {
            // let the fish die
            _localEnv.removeFish(this);
        }
    }

    /**
     * Return a clone this instance.
     *
     * @return clone
     */
    public Object clone() {
        Object clone;
        try {
            clone = super.clone();
        }
        catch (CloneNotSupportedException e) {
            throw new RuntimeException("AFish.clone failed");
        }
        return clone;
    }

    /**
     * Return a string representation of this fish.
     *
     * @return string representation
     */
    public String toString() {
        String className = this.getClass().getName();
        return className.substring(className.lastIndexOf('.') + 1);
    }

    /**
     * Execute the movement part of a simulation step.
     */
    protected abstract void move();

    /**
     * Draw the fish. The Graphics2D object has been translated, rotated and scaled so that the origin is in
     * the center of the fish. The fish should just be drawn from (-0.5, -0.5) to (0.5, 0.5).
     *
     * @param g    graphics object to drawFish on.
     * @param comp the component to drawFish on
     */
    public void paint(Graphics2D g, Component comp) {
        _fishDisplay.draw(g, comp, getColor());
    }

    /**
     * Get the fish's color.
     *
     * @return color of the fish
     */
    public Color getColor() {
        return _fishColor;
    }

    /**
     * This method is called whenever the observed object is changed. The fish executes the lambda that is passed
     * in with the fish itself as parameter.
     *
     * @param o   the observable object
     * @param arg lambda to execute
     */
    public final void update(Observable o, Object arg) {
        ((ILambda) arg).apply(new FishApplyParams(this, _localEnv));
    }

    /**
     * Set probability of dying.
     *
     * @param p the probability
     */
    public void setProbOfDying(double p) {
        _probOfDying = p;
    }

    /**
     * Set probability of breeding.
     *
     * @param p the probability
     */
    public void setProbOfBreeding(double p) {
        _probOfBreeding = p;
    }
}
