package view;

import controller.IDisplayAdapter;
import controller.IEnvAdapter;

import javax.swing.*;
import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;

/**
 * Panel to display the environment.
 *
 * @author Mathias Ricken
 */
public class DisplayPanel extends JPanel implements Scrollable, DisplayViewport.Pannable {
    // Class constants
    private static final int MIN_CELL_SIZE = 8;
    private static final int DEFAULT_CELL_SIZE = 32;

    /**
     * Display adapter.
     */
    IDisplayAdapter _displayAdapter;

    /**
     * Cell size.
     */
    int _cellSize = DEFAULT_CELL_SIZE;

    /**
     * Origin column.
     */
    double _originX;

    /**
     * Origin row.
     */
    double _originY;

    /**
     * Mouse adapter for editing.
     */
    MouseAdapter _mouseAdapter;

    /**
     * State of tool tips.
     */
    private boolean _toolTipsEnabled;

    /**
     * Environment adapter.
     */
    IEnvAdapter _envAdapter;

    /**
     * Make a new display panel.
     *
     * @param da display adapter to use
     * @param ea environment adapter to use
     */
    public DisplayPanel(IDisplayAdapter da, final IEnvAdapter ea) {
        _displayAdapter = da;
        _envAdapter = ea;
        _mouseAdapter = new MouseAdapter() {
            public void mousePressed(MouseEvent evt) {
                double x = ((double) evt.getPoint().x) / _cellSize + _originX;
                double y = ((double) evt.getPoint().y) / _cellSize + _originY;

                ea.edit(new Point.Double(x, y),evt.getButton());
                revalidate();
            }
        };
        setToolTipsEnabled(true);
        revalidate();
    }

    /**
     * Enable or disable the mouse adapter for editing.
     *
     * @param enable true to enable
     */
    public void enableMouseAdapter(boolean enable) {
        if (enable) {
            addMouseListener(_mouseAdapter);
        }
        else {
            removeMouseListener(_mouseAdapter);
        }
    }

    /**
     * Enable or disable showing of tooltip giving information about
     * the environment object beneath the mouse.
     *
     * @param flag whether to enable/disable tool tips
     */
    public void setToolTipsEnabled(boolean flag) {
        if (flag) {
            ToolTipManager.sharedInstance().registerComponent(this);
        }
        else {
            ToolTipManager.sharedInstance().unregisterComponent(this);
        }
        _toolTipsEnabled = flag;
    }

    /**
     * Given a MouseEvent, determine what text to place in the floating tool tip when the
     * the mouse hovers over this location.  If the mouse is over a valid environment cell.
     * we provide some information about the cell and its contents.  This method is
     * automatically called on mouse-moved events since we register for tool tips.
     *
     * @param evt the MouseEvent in question
     * @return the tool tip string for this location
     */
    public String getToolTipText(MouseEvent evt) {
        double x = ((double) evt.getPoint().x) / _cellSize + _originX;
        double y = ((double) evt.getPoint().y) / _cellSize + _originY;
        return _envAdapter.getToolTipText(new Point.Double(x, y));
    }

    /**
     * Zoom in.
     */
    public void zoomIn() {
        double oldOriginX = _originX;
        double oldOriginY = _originY;
        JViewport vp = (JViewport) getParent();
        Point pt = vp.getViewPosition();
        pt.x = (int) (((double) pt.x) / _cellSize + _originX);
        pt.y = (int) (((double) pt.y) / _cellSize + _originY);

        _cellSize *= 2;
        setCorner(pt.x, pt.y);
        _originX = oldOriginX;
        _originY = oldOriginY;
        revalidate();
    }

    /**
     * Zoom out.
     */
    public void zoomOut() {
        double oldOriginX = _originX;
        double oldOriginY = _originY;
        JViewport vp = (JViewport) getParent();
        Point pt = vp.getViewPosition();
        pt.x = (int) (((double) pt.x) / _cellSize + _originX);
        pt.y = (int) (((double) pt.y) / _cellSize + _originY);

        _cellSize = Math.max(MIN_CELL_SIZE, _cellSize / 2);
        setCorner(pt.x, pt.y);
        _originX = oldOriginX;
        _originY = oldOriginY;
        revalidate();
    }

    /**
     * Make field (x,y) visible in top left corner.
     * @param x x-coordinate
     * @param y y-coordinate
     */
    public void setCorner(int x, int y) {
        DisplayViewport vp = (DisplayViewport) getParent();
        vp.oldSetViewPosition(new Point((int) (x - _originX) * _cellSize, (int) (y - _originY) * _cellSize));
    }

    /**
     * Returns the desired size of the display, for use by layout manager.
     *
     * @return preferred size
     */
    public Dimension getPreferredSize() {
        Dimension s = _displayAdapter.getDisplaySize();
        return new Dimension(s.width * _cellSize, s.height * _cellSize);
    }

    /**
     * Returns the minimum size of the display, for use by layout manager.
     *
     * @return minimum size
     */
    public Dimension getMinimumSize() {
        Dimension s = _displayAdapter.getDisplaySize();
        return new Dimension(s.width * MIN_CELL_SIZE, s.height * MIN_CELL_SIZE);
    }

    public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation, int direction) {
        return _cellSize;
    }

    public int getScrollableBlockIncrement(Rectangle visibleRect, int orientation, int direction) {
        if (orientation == SwingConstants.VERTICAL) {
            return (int) (visibleRect.height * .9);
        }
        else {
            return (int) (visibleRect.width * .9);
        }
    }

    public boolean getScrollableTracksViewportWidth() {
        return false;
    }

    public boolean getScrollableTracksViewportHeight() {
        return false;
    }

    public Dimension getPreferredScrollableViewportSize() {
        return new Dimension(420, 420);
    }

    /**
     * Paint this component.
     *
     * @param g the Graphics object to use to render this component
     */
    public void paintComponent(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        super.paintComponent(g2);

        Rectangle curClip = g2.getClipBounds();
        Graphics2D envGraphics = (Graphics2D) g2.create();
        AffineTransform oldTransform = envGraphics.getTransform();
        int left = getInsets().left, top = getInsets().top;

        // transform to model coordinates
        int x1 = (curClip.x - left) / _cellSize;
        int y1 = (curClip.y - top) / _cellSize;
        int x2 = (curClip.x + curClip.width - left + _cellSize - 1) / _cellSize;
        int y2 = (curClip.y + curClip.height - top + _cellSize - 1) / _cellSize;

        // translate and scale so that (0,0) corresponds to the field (x1,y1) in model coordinates
        // and each field in model coordinates is MODEL_CELL_SIZE wide
        envGraphics.translate(x1 * _cellSize + left, y1 * _cellSize + top);
        envGraphics.scale(_cellSize, _cellSize);
        envGraphics.setStroke(new BasicStroke(1.0f / _cellSize));
        _displayAdapter.draw(envGraphics, this, new Point.Double(x1 + (int) _originX, y1 + (int) _originY), new Point.Double(x2 + (int) _originX, y2 + (int) _originY));

        envGraphics.setTransform(oldTransform);
    }

    /**
     * Pan the panel by the specified amount of pixels.
     *
     * @param dx horizontal pan in pixels
     * @param dy vertical pan in pixels
     */
    public void pan(double dx, double dy) {
        _originX = (int) (_originX + dx / _cellSize);
        _originY = (int) (_originY + dy / _cellSize);
    }

    /**
     * Return the current cell size.
     *
     * @return cell size
     */
    public int getCellSize() {
        return _cellSize;
    }

    /**
     * Reset the pan.
     */
    public void resetPan() {
        Point.Double center = _displayAdapter.getViewPosition(new Point.Double(0, 0));
        _originX = -center.x;
        _originY = -center.y;
        repaint();
    }

    /**
     * Get the tool tip text for panning.
     *
     * @return pan tool tip
     */
    public String getPanTipText() {
        JViewport vp = (JViewport) getParent();
        Point pt = vp.getViewPosition();
        int x = (int) (((double) pt.x) / _cellSize + _originX);
        int y = (int) (((double) pt.y) / _cellSize + _originY);
        return "(" + x + "," + y + ")";
    }
}
