package sysModel.env;

import controller.IScrollAdapter;
import junit.framework.TestCase;
import lrs.IAlgo;
import lrs.LRStruct;
import lrs.visitor.GetLength;
import lrs.visitor.Remove;
import model.ILambda;
import model.fish.GenericFish;
import sysModel.ICmdFactory;
import sysModel.NoOpLambda;
import sysModel.fish.AFish;
import sysModel.fish.IFishFactory;
import sysModel.parser.Lexer;

import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.util.LinkedList;

/**
 * Implementation of a square unbounded environment.
 *
 * @author Mathias G. Ricken
 */
public class UnboundedEnv extends ASquareEnv {

    /**
     * Concrete local environment for the square unbounded environment.
     */
    protected class LocalEnvironment implements ISquareLocalEnvironment {
        /**
         * Location.
         */
        Location _loc;

        /**
         * Direction.
         */
        Direction _dir;

        /**
         * State.
         */
        ILocalEnvState _state = EmptyLocalEnvState.Singleton;

        /**
         * Lambda to execute a move.
         */
        private class MoveLambda implements ILambda {
            /// target direction
            private Direction _newDir;
            /// target location
            private Location _newLoc;

            /**
             * Constructor.
             *
             * @param le target local environment
             */
            public MoveLambda(LocalEnvironment le) {
                _newLoc = makeLocation(le._loc.getX(), le._loc.getY());
                _newDir = makeDirection(le._dir);
            }

            /**
             * Execute the move.
             *
             * @param param not used
             * @return null
             */
            public Object apply(Object param) {
                // execute the movement
                _loc = _newLoc;
                _dir = _newDir;

                // deactivate all lambdas
                deactivateMoveLambdas();
                return null;
            }
        }

        /**
         * Construct a new local environment.
         *
         * @param loc location
         * @param dir direction
         */
        public LocalEnvironment(Location loc, Direction dir) {
            _loc = loc;
            _dir = dir;
        }

        /**
         * Accessor for the location.
         *
         * @return location
         */
        public Location location() {
            return _loc;
        }

        /**
         * Accessor for the direction.
         *
         * @return direction
         */
        public Direction direction() {
            return _dir;
        }

        /**
         * Make local environment in forward direction. Do not block yourself.
         *
         * @return new local environment in forward direction
         */
        private ILocalEnv makeMoveFwdLocalEnv() {
            // remove this local environment to prevent collision with itself
            _localEnvList.execute(Remove.Singleton, this);
            ILocalEnv le = makeLocalEnv(_loc.getNeighbor(_dir), _dir);
            // add this local environment back in
            _localEnvList.insertFront(LocalEnvironment.this);
            return le;
        }

        /**
         * Attempt to move the fish forward, which may or may not be successful.
         * The behavior in each case is defined by the visitor:
         * - If the move cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
         * - If the move can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
         * executed to actually move the fish to the target location of this move. The ILambda ignores the input
         * parameter and returns null.
         *
         * @param fish       AFish to move
         * @param blockedCmd lambda to apply if blocked
         * @param openCmd    lambda to apply if open
         * @return return value of lambda executed
         */
        public Object tryMoveFwd(AFish fish, final ILambda blockedCmd, final ILambda openCmd) {
            // TODO: PART3
            return null;
        }

        /**
         * Attempt to breed the fish forward, which may or may not be successful.
         * The behavior in each case is defined by the visitor:
         * - If the breeding cannot be executed, the blockedCmd lambda is applied. The parameter is not used and set to null.
         * - If the breeding can be executed, the openCmd lambda is applied. The parameter is an ILambda that can to be
         * executed to actually move the fish to the target location of this breeding. The ILambda ignores the input
         * parameter and returns null.
         *
         * @param fish       AFish to move
         * @param blockedCmd lambda to apply if blocked
         * @param openCmd    lambda to apply if open
         * @return return value of lambda executed
         */
        public Object tryBreedFwd(final AFish fish, final ILambda blockedCmd, final ILambda openCmd) {
            // TODO: PART3
            return null;
        }

        /**
         * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
         *
         * @param fish AFish to drawFish
         * @param g    graphics object to drawFish on
         * @param comp component to drawFish on
         */
        public void drawFish(AFish fish, Graphics2D g, Component comp) {
            final double centerX = Math.floor(_loc.getX()) + 1.0 / 2;
            final double centerY = Math.floor(_loc.getY()) + 1.0 / 2;

            // save transformation
            AffineTransform oldTransform = g.getTransform();
            // translate to center of field
            g.translate(centerX, centerY);

            // set up the correct rotation
            _dir.rotateGraphics(g);

            // makeDrawCmd the fish
            fish.paint(g, comp);

            // restore transformation
            g.setTransform(oldTransform);
        }

        /**
         * Turn the fish radians to the right.
         *
         * @param fish    AFish to turn
         * @param radians radians to turn
         */
        public void turnRight(AFish fish, double radians) {
            _dir.turnRight(radians);
        }

        /**
         * Remove the fish from the environment.
         *
         * @param fish AFish to remove
         */
        public void removeFish(AFish fish) {
            UnboundedEnv.this.removeFish(this);
        }

        /**
         * Execute a visitor on this local environment.
         *
         * @param visitor visitor to execute
         * @param param   visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object execute(final ILocalEnvVisitor visitor, final Object param) {
            return _state.execute(this, visitor, param);
        }

        /**
         * String representation of the local environment.
         * Should be "(x, y) (dx, dy)".
         *
         * @return string representation
         */
        public String toString() {
            return _loc.toString() + " " + _dir.toString();
        }

        /**
         * Change the state of this local environment.
         *
         * @param state new state
         */
        public void setState(ILocalEnvState state) {
            _state = state;
        }
    }

    /**
     * Size of the pan area.
     */
    protected final int PAN_SIZE = 2000;

    /**
     * Center of the pan area.
     */
    protected final Point.Double PAN_CENTER = new Point.Double(PAN_SIZE / 2, PAN_SIZE / 2);

    /**
     * List of local environments in this global environment.
     */
    private LRStruct _localEnvList;

    /**
     * Construct a new square unbounded environment.
     *
     * @param cmdFactory command factory to use
     */
    public UnboundedEnv(ICmdFactory cmdFactory) {
        super(cmdFactory);
        _localEnvList = new LRStruct();
    }

    /**
     * Add the fish to the global environment.
     *
     * @param localEnv local environment
     * @param fish     fish to add
     */
    protected void addFishToInternalData(ILocalEnv localEnv, AFish fish) {
        _localEnvList.insertFront(localEnv);
    }

    /**
     * Remove the fish from the global environment.
     *
     * @param localEnv local environment
     */
    protected void removeFishFromInternalData(ILocalEnv localEnv) {
        _localEnvList.execute(Remove.Singleton, localEnv);
    }

    /**
     * Create a local environment for the position.
     *
     * @param p position
     * @return local environment
     */
    public ILocalEnv makeLocalEnv(Point.Double p) {
        return makeLocalEnv(makeLocation(p.getX(), p.getY()), makeDirection());

    }

    /**
     * Create a local environment for the position.
     *
     * @param loc location
     * @param dir direction
     * @return local environment
     */
    protected ISquareLocalEnvironment makeLocalEnv(final Location loc, final Direction dir) {
        return (ISquareLocalEnvironment) _localEnvList.execute(new IAlgo() {
            /**
             * Operates on a non-empty LRStruct host, given an input object.
             *
             * @param host a non-empty LRStruct.
             * @param inp  input object needed by this IAlgo.
             * @return an appropriate output object.
             */
            public Object nonEmptyCase(LRStruct host, Object inp) {
                LocalEnvironment localEnv = (LocalEnvironment) host.getFirst();
                if (localEnv.location().inField(loc)) {
                    return localEnv;
                }
                else {
                    return host.getRest().execute(this, inp);
                }
            }

            /**
             * Operates on an empty LRStruct host, given an input object.
             *
             * @param host an empty LRStruct.
             * @param inp  input object needed by this IAlgo.
             * @return an appropriate output object.
             */
            public Object emptyCase(LRStruct host, Object inp) {
                return new LocalEnvironment(loc, dir);
            }
        }, null);
    }

    /**
     * Factory method for parsing a stream of tokens and creating a global environment from it.
     *
     * @param l lexer to use
     * @return new global environment
     */
    protected AGlobalEnv parseEnvironment(Lexer l) {
        UnboundedEnv env = new UnboundedEnv(_cmdFactory);
        env.parseFish(l);
        return env;
    }

    /**
     * Get the environment settings class.
     *
     * @return environment settings class
     */
    public AEnvFactory makeEnvFactory() {
        return new AEnvFactory() {
            public AGlobalEnv create() {
                return new UnboundedEnv(_cmdFactory);
            };
            public String toString() {
                return UnboundedEnv.class.getName();
            }
        };
    }

    /**
     * Print file header.
     *
     * @param pw PrintWriter to use
     */
    protected void printHeader(java.io.PrintWriter pw) {
        pw.println(this.getClass().getName());
    }

    /**
     * Get size of the display.
     *
     * @return size of the display in model coordinate units.
     */
    public Dimension getDisplaySize() {
        return new Dimension(PAN_SIZE, PAN_SIZE);
    }

    /**
     * The action to be executed if the display should return home.
     *
     * @param sa scroll adapter
     */
    public void returnHome(final IScrollAdapter sa) {
        sa.setCorner((int) PAN_CENTER.x, (int) PAN_CENTER.y);
        sa.resetScrolling();
    }

    /**
     * Ask the model where to scroll, given where the user has scrolled.
     * If the environment just acts like a normal panal, it should return pos without modification.
     * If the environment recenters, it should return a position in the middle of the pan area.
     * All coordinates are in model coordinate units.
     *
     * @param pos position where the user scrolled to
     * @return position where the environment wants the view to be
     * @see controller.IDisplayAdapter#getPanDelta
     */
    public Point.Double getViewPosition(Point.Double pos) {
        // the panel always gets recentered after moving it, so return the center position
        return PAN_CENTER;
    }

    /**
     * Ask the model how much to pan, given where the user scrolled.
     * If the environment just acts like a normal panal, it should return (0,0).
     * If the environment recenters, it should return delta without modification.
     * All coordinates are in model coordinate units.
     *
     * @param delta how far the user scrolled
     * @return how far the panel should scroll
     * @see controller.IDisplayAdapter#getViewPosition
     */
    public Point.Double getPanDelta(Point.Double delta) {
        // we want the panel to keep track of the position, so return the delta as pan value
        return delta;
    }

    /*****************************************************************************************************************
     * Tests follow
     *****************************************************************************************************************/

    /**
     * Test cases for UnboundedEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_UnboundedEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private UnboundedEnv _env;
        private IFishFactory _fishFactory;

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new UnboundedEnv(_cmdFactory);

            _fishFactory = new IFishFactory() {
                /**
                 * Create a new fish.
                 *
                 * @return new fish
                 */
                public AFish createFish() {
                    return new GenericFish(Color.RED);
                }
            };
        }

        /**
         * Test addFishToInternalData.
         */
        public void testAddFish() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFishToInternalData(lTop, fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFishToInternalData(lBottom, fBottom);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
        }

        /**
         * Test editFish.
         */
        public void testEditFish() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFishToInternalData(l, f);
            Direction d = l.direction();

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(0.0, d.getAngle(), 0.01);

            ILambda lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(Math.PI, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(0, d.getAngle(), 0.01);
            assertEquals(_delete, lambda);
        }

        /**
         * Test getViewPosition.
         */
        public void testGetViewPosition() {
            Point.Double panCenter = _env.PAN_CENTER;

            assertTrue(_env.getViewPosition(new Point.Double(0, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(1.0, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(1.2, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.3)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(-2.5, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(-3.0, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, -2.5)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, -3.0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(2.0, 1.0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(-4.0, -2.3)).equals(panCenter));
        }

        /**
         * Test getPanDelta.
         */
        public void testGetPanDelta() {
            assertTrue(_env.getPanDelta(new Point.Double(0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.0, 0)).equals(new Point.Double(1.0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.2, 0)).equals(new Point.Double(1.2, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.0)).equals(new Point.Double(0, 1.0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.3)).equals(new Point.Double(0, 1.3)));
            assertTrue(_env.getPanDelta(new Point.Double(-2.5, 0)).equals(new Point.Double(-2.5, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-3.0, 0)).equals(new Point.Double(-3.0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -2.5)).equals(new Point.Double(0, -2.5)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -3.0)).equals(new Point.Double(0, -3.0)));
            assertTrue(_env.getPanDelta(new Point.Double(2.0, 1.0)).equals(new Point.Double(2.0, 1.0)));
            assertTrue(_env.getPanDelta(new Point.Double(-4.0, -2.3)).equals(new Point.Double(-4.0, -2.3)));
        }
    }

    /**
     * Test cases for UnboundedEnv.LocalEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_UnboundedEnv_LocalEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private UnboundedEnv _env;

        private static class SuccessException extends RuntimeException {
            public SuccessException() {
                super();
            }
        }

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new UnboundedEnv(_cmdFactory);
        }

        /**
         * Test local environment's execute.
         */
        public void testExecute() {
            ILocalEnv l = _env.makeLocalEnv(new Point.Double(1, 1));

            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty --");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch(SuccessException e) { }

            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);

            ILocalEnv l2 = _env.makeLocalEnv(new Point.Double(1, 1));
            try {
                l2.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty --");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been falled --");
            }
            catch(SuccessException e) { }

            ILocalEnv l3 = _env.makeLocalEnv(new Point.Double(1.4, 1.6));
            try {
                l3.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty --");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been falled --");
            }
            catch(SuccessException e) { }

            ILocalEnv l4 = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty --");
                    }
                }, null);
                fail("emptyCase should have been falled --");
            }
            catch(SuccessException e) { }

            GenericFish f4 = new GenericFish(Color.RED);
            f4.setLocalEnvironment(l4);
            _env.addFish(l4, f4);
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty --");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been falled --");
            }
            catch(SuccessException e) { }

        }

        /**
         * Test local environment's tryMoveFwd.
         */
        public void testTryMoveFwd() {
            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);

            // move lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(456), i);

            // move lTop --> open, don't move
            i = ((Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(123);
                }
            }));
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(123), i);

            // move lBottom into lTop --> blocked
            i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(789), i);

            // move lTop --> open, move
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, make move
                    ((ILambda) param).apply(null);
                    return new Integer(111);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(111), i);

            // move lBottom --> open
            i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(222);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(222), i);

            // move lTop --> open, don't move
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(333);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(333), i);

            // turn and move lTop --> open, don't move
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(444);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(444), i);
        }

        /**
         * Test local environment's tryBreedFwd.
         */
        public void testTryBreedFwd() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // breed lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> open, don't breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            assertEquals(3, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // turn and breed lTop --> open, don't breed
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(3, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // turn and breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(4, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // turn and breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked --");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(4, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
        }

        /**
         * Test local environment's turnRight.
         */
        public void testTurnRight() {
            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            Direction d = l.direction();

            assertEquals(0.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(0, d.getAngle(), 0.01);
        }

        /**
         * Test local environment's removeFish.
         */
        public void testRemoveFish() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            lTop.removeFish(fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            lBottom.removeFish(fBottom);

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
        }


        /**
         * Test to make sure only one move lambda can be executed.
         */
        public void testOnlyOneMove() {
            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(5.0, 5.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);

            final LinkedList lambdas = new LinkedList();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            ((ILambda) lambdas.get(0)).apply(null);
            Location loc = l.location();
            assertTrue("Should have moved -- ",loc.same(_env.makeLocation(5.0, 4.0)));

            ((ILambda) lambdas.get(1)).apply(null);
            loc = l.location();
            assertTrue("Should not have moved, move lambda was not deactivated -- ",loc.same(_env.makeLocation(5.0, 4.0)));

            lambdas.clear();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open --");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    lambdas.add(param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            ((ILambda) lambdas.get(1)).apply(null);
            loc = l.location();
            assertTrue("Should have moved -- ",loc.same(_env.makeLocation(6.0, 4.0)));

            ((ILambda) lambdas.get(0)).apply(null);
            loc = l.location();
            assertTrue("Should not have moved, move lambda was not deactivated -- ",loc.same(_env.makeLocation(6.0, 4.0)));
        }
    }
}