package sysModel.env;

import sysModel.ICmdFactory;
import sysModel.NoOpLambda;
import sysModel.fish.AFish;
import sysModel.fish.IFishFactory;

import javax.swing.*;

import junit.framework.TestCase;
import model.ILambda;
import model.fish.GenericFish;

import java.awt.geom.Point2D;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.util.LinkedList;

/**
 * Implementation of a square wrapping environment.
 *
 * @author Mathias G. Ricken
 */
public class WrappingEnv extends BoundedEnv {

    /**
     * Factory method for overridden Location.
     * @param x x coordinate
     * @param y y coordinate
     * @return new WrappingEnv.Location at (x, y)
     */
    public ASquareEnv.Location makeLocation(double x, double y) {
        return new Location(x,y);
    }

    /**
     * Overridden location class that does wrapping.
     *
     * @author Mathias G. Ricken
     */
    public class Location extends ASquareEnv.Location {
        /**
         * Ctor for location.
         * @param x x position
         * @param y y position
         */
        public Location(double x, double y) {
            super(x, y);
        }

        /**
         * Return the location of a neighbor in the given direction.
         *
         * @param dir the direction of the neighbor to be returned
         * @return neighbor in that direction
         */
        public ASquareEnv.Location getNeighbor(Direction dir) {
            double newX = getX() + dir.getDeltaX();
            double newY = getY() + dir.getDeltaY();
            if (newX < 0) {
                newX = _width + newX;
            }
            else if (newX >= _width) {
                newX = newX - _width;
            }
            if (newY < 0) {
                newY = _height + newY;
            }
            else if (newY >= _height) {
                newY = newY - _height;
            }
            return makeLocation(newX,newY);
        }
    }

    /**
     * Construct a new square wrapping environment.
     * Does not set this object up for actual use.
     * Note: This constructor needs to exist and be public for the "environment selection"
     * dialog to work.
     *
     * @param cmdFactory command factory to use
     */
    public WrappingEnv(ICmdFactory cmdFactory) {
        super(cmdFactory);
    }

    /**
     * Construct a new square wrapping environment.
     *
     * @param cmdFactory command factory to use
     * @param width      width of environment
     * @param height     height of environment
     */
    public WrappingEnv(ICmdFactory cmdFactory, int width, int height) {
        super(cmdFactory,width,height);
    }

    /**
     * Get the environment settings class.
     *
     * @return environment settings class
     */
    public AEnvFactory makeEnvFactory() {
        return new AEnvFactory() {
            private JTextField _rowField;
            private JTextField _colField;

            {
                setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
                add(new JLabel("rows: "));
                add(_rowField = new JTextField("10"));
                add(new JLabel("  cols: "));
                add(_colField = new JTextField("10"));
            }

            public AGlobalEnv create() {
                return new WrappingEnv(WrappingEnv.this._cmdFactory, Integer.parseInt(_colField.getText()), Integer.parseInt(_rowField.getText()));
            };
            public String toString() {
                return WrappingEnv.class.getName();
            }
        };
    }

    /*****************************************************************************************************************
     * Tests follow
     *****************************************************************************************************************/

    /**
     * Test cases for WrappingEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_WrappingEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private WrappingEnv _env;
        private IFishFactory _fishFactory;

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new WrappingEnv(_cmdFactory, 10, 10);

            _fishFactory = new IFishFactory() {
                /**
                 * Create a new fish.
                 *
                 * @return new fish
                 */
                public AFish createFish() {
                    return new GenericFish(Color.RED);
                }
            };
        }

        int countNonEmpty() {
            int count = 0;
            WrappingEnv.IField[][] map = _env._fieldMap;
            for (int i = 0; i < map.length; i++) {
                WrappingEnv.IField[] iFields = map[i];
                for (int j = 0; j < iFields.length; j++) {
                    WrappingEnv.IField iField = iFields[j];
                    count += ((Integer) iField.execute(new WrappingEnv.IFieldVisitor() {
                        public Object emptyCase(WrappingEnv.EmptyField host, Object param) {
                            return new Integer(0);
                        }

                        public Object nonEmptyCase(WrappingEnv.NonEmptyField host, Object param) {
                            return new Integer(1);
                        }
                    }, null)).intValue();
                }
            }
            return count;
        }

        /**
         * Test addFishToInternalData.
         */
        public void testAddFish() {
            assertEquals(0, countNonEmpty());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFishToInternalData(lTop, fTop);

            assertEquals(1, countNonEmpty());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFishToInternalData(lBottom, fBottom);

            assertEquals(2, countNonEmpty());
        }

        /**
         * Test editFish.
         */
        public void testEditFish() {
            assertEquals(0, countNonEmpty());

            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFishToInternalData(l, f);
            WrappingEnv.Direction d = l.direction();

            assertEquals(1, countNonEmpty());
            assertEquals(0.0, d.getAngle(), 0.01);

            ILambda lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, countNonEmpty());
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, countNonEmpty());
            assertEquals(Math.PI, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(1, countNonEmpty());
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            assertEquals(NoOpLambda.instance(), lambda);

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(0, countNonEmpty());
            assertEquals(0, d.getAngle(), 0.01);
            assertEquals(_delete, lambda);
        }

        /**
         * Test getViewPosition.
         */
        public void testGetViewPosition() {
            assertTrue(_env.getViewPosition(new Point.Double(0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(1.0, 0)).equals(new Point.Double(1.0, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(1.2, 0)).equals(new Point.Double(1.2, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.0)).equals(new Point.Double(0, 1.0)));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.3)).equals(new Point.Double(0, 1.3)));
            assertTrue(_env.getViewPosition(new Point.Double(-2.5, 0)).equals(new Point.Double(-2.5, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(-3.0, 0)).equals(new Point.Double(-3.0, 0)));
            assertTrue(_env.getViewPosition(new Point.Double(0, -2.5)).equals(new Point.Double(0, -2.5)));
            assertTrue(_env.getViewPosition(new Point.Double(0, -3.0)).equals(new Point.Double(0, -3.0)));
            assertTrue(_env.getViewPosition(new Point.Double(2.0, 1.0)).equals(new Point.Double(2.0, 1.0)));
            assertTrue(_env.getViewPosition(new Point.Double(-4.0, -2.3)).equals(new Point.Double(-4.0, -2.3)));
        }

        /**
         * Test getPanDelta.
         */
        public void testGetPanDelta() {
            assertTrue(_env.getPanDelta(new Point.Double(0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.2, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.3)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-2.5, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-3.0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -2.5)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -3.0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(2.0, 1.0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-4.0, -2.3)).equals(new Point.Double(0, 0)));
        }
    }

    /**
     * Test cases for WrappingEnv.LocalEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_WrappingEnv_LocalEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private WrappingEnv _env;

        private static class SuccessException extends RuntimeException {
            public SuccessException() {
                super();
            }
        }

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new WrappingEnv(_cmdFactory, 10, 10);
        }

        /**
         * Test local environment's execute.
         */
        public void testExecute() {
            ILocalEnv l = _env.makeLocalEnv(new Point.Double(1, 1));

            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch(SuccessException e) { }


            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            l.setState(NonEmptyLocalEnvState.Singleton);
            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }

            ILocalEnv l2 = _env.makeLocalEnv(new Point.Double(1, 1));
            try {
                l2.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }

            ILocalEnv l3 = _env.makeLocalEnv(new Point.Double(1.4, 1.6));
            try {
                l3.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }

            ILocalEnv l4 = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch(SuccessException e) { }

            GenericFish f4 = new GenericFish(Color.RED);
            f4.setLocalEnvironment(l4);
            _env.addFish(l4, f4);
            l4.setState(NonEmptyLocalEnvState.Singleton);
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ILocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ILocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch(SuccessException e) { }
        }

        /**
         * Test local environment's tryMoveFwd.
         */
        public void testTryMoveFwd() {
            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);
            lTop.setState(NonEmptyLocalEnvState.Singleton);

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);
            lBottom.setState(NonEmptyLocalEnvState.Singleton);

            // move lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // move lTop --> open, don't move
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            // move lBottom into lTop --> blocked
            i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(789), i);

            // move lTop --> open, move
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, make move
                    ((ILambda) param).apply(null);
                    return new Integer(111);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(111), i);

            // move lBottom --> open
            i = (Integer) lBottom.tryMoveFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(222);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(222), i);

            // move lTop into edge, wrap --> open
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(333);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(333), i);

            // turn and move lTop --> open
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryMoveFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(444);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(444), i);
        }

        int countNonEmpty() {
            int count = 0;
            WrappingEnv.IField[][] map = _env._fieldMap;
            for (int i = 0; i < map.length; i++) {
                WrappingEnv.IField[] iFields = map[i];
                for (int j = 0; j < iFields.length; j++) {
                    WrappingEnv.IField iField = iFields[j];
                    count += ((Integer) iField.execute(new WrappingEnv.IFieldVisitor() {
                        public Object emptyCase(WrappingEnv.EmptyField host, Object param) {
                            return new Integer(0);
                        }

                        public Object nonEmptyCase(WrappingEnv.NonEmptyField host, Object param) {
                            return new Integer(1);
                        }
                    }, null)).intValue();
                }
            }
            return count;
        }

        /**
         * Test local environment's tryBreedFwd.
         */
        public void testTryBreedFwd() {
            assertEquals(0, countNonEmpty());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);
            lTop.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(1, countNonEmpty());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);
            lBottom.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(2, countNonEmpty());

            // breed lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> open, don't breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            assertEquals(2, countNonEmpty());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(123), i);

            assertEquals(3, countNonEmpty());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(456), i);

            // turn and breed lTop --> open, don't breed
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(3, countNonEmpty());

            // turn and breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --",param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",DeactivatableLambda.class,param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(4, countNonEmpty());

            // turn and breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new ILambda() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",new Integer(789), i);

            assertEquals(4, countNonEmpty());
        }

        /**
         * Test local environment's turnRight.
         */
        public void testTurnRight() {
            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            l.setState(NonEmptyLocalEnvState.Singleton);
            WrappingEnv.Direction d = l.direction();

            assertEquals(0.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(0, d.getAngle(), 0.01);
        }

        /**
         * Test local environment's removeFish.
         */
        public void testRemoveFish() {
            assertEquals(0, countNonEmpty());

            ILocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);
            lTop.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(1, countNonEmpty());

            ILocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);
            lBottom.setState(NonEmptyLocalEnvState.Singleton);

            assertEquals(2, countNonEmpty());

            lTop.removeFish(fTop);

            assertEquals(1, countNonEmpty());

            lBottom.removeFish(fBottom);

            assertEquals(0, countNonEmpty());
        }

        /**
         * Test to make sure only one move lambda can be executed.
         */
        public void testOnlyOneMove() {
            ISquareLocalEnvironment l = (ISquareLocalEnvironment)_env.makeLocalEnv(new Point.Double(5.0, 5.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            l.setState(NonEmptyLocalEnvState.Singleton);

            final LinkedList lambdas = new LinkedList();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    lambdas.add(param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    lambdas.add(param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            ((ILambda) lambdas.get(0)).apply(null);
            Location loc = (Location)l.location();
            assertTrue(loc.same(_env.makeLocation(5.0, 4.0)));

            ((ILambda) lambdas.get(1)).apply(null);
            loc = (Location)l.location();
            assertTrue(loc.same(_env.makeLocation(5.0, 4.0)));

            lambdas.clear();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    lambdas.add(param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new ILambda() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new ILambda() {
                public Object apply(Object param) {
                    lambdas.add(param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            ((ILambda) lambdas.get(1)).apply(null);
            loc = (Location)l.location();
            assertTrue(loc.same(_env.makeLocation(6.0, 4.0)));

            ((ILambda) lambdas.get(0)).apply(null);
            loc = (Location)l.location();
            assertTrue(loc.same(_env.makeLocation(6.0, 4.0)));
        }

    }

    /**
     * Test cases for WrappingEnv.Location.
     *
     * @author Mathias Ricken
     */
    public static class Test_WrappingEnv_Location extends TestCase {
        private ICmdFactory _cmdFactory;
        private WrappingEnv _env;

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() {
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ILocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };

            _env = new WrappingEnv(_cmdFactory,50,50);
        }

        /**
         * Test getters, setters, and ctor.
         */
        public void testGetSet() {
            ASquareEnv.Location l = _env.makeLocation(5, 7);
            assertEquals(5.0, l.getX(), 0.01);
            assertEquals(7.0, l.getY(), 0.01);

            l.setX(11);
            assertEquals(11.0, l.getX(), 0.01);
            assertEquals(7.0, l.getY(), 0.01);

            l.setY(13);
            assertEquals(11.0, l.getX(), 0.01);
            assertEquals(13.0, l.getY(), 0.01);
        }

        /**
         * Test same.
         */
        public void testSame() {
            ASquareEnv.Location l = _env.makeLocation(17, 23);
            ASquareEnv.Location l1 = _env.makeLocation(17, 23);
            ASquareEnv.Location l2 = _env.makeLocation(31, 37);
            assertEquals(true, l.same(l1));
            assertEquals(false, l.same(l2));
            assertEquals(true, l1.same(l));
            assertEquals(false, l2.same(l));
            assertEquals(false, l2.same(l1));
            assertEquals(false, l1.same(l2));
        }

        /**
         * Test inField.
         */
        public void testInField() {
            ASquareEnv.Location l = _env.makeLocation(17, 23);
            ASquareEnv.Location l1 = _env.makeLocation(17, 23);
            ASquareEnv.Location l2 = _env.makeLocation(17.13, 23.53);
            ASquareEnv.Location l3 = _env.makeLocation(17.49, 23.77);
            ASquareEnv.Location l4 = _env.makeLocation(31, 37);
            ASquareEnv.Location l5 = _env.makeLocation(31.81, 37.51);
            assertEquals(true, l.inField(l1));
            assertEquals(true, l1.inField(l));
            assertEquals(true, l.inField(l2));
            assertEquals(true, l2.inField(l));
            assertEquals(true, l.inField(l3));
            assertEquals(true, l3.inField(l));
            assertEquals(false, l.inField(l4));
            assertEquals(false, l4.inField(l));
            assertEquals(false, l.inField(l5));
            assertEquals(false, l5.inField(l));
        }

        /**
         * Test getNeighbor.
         */
        public void testNeighbor() {
            ASquareEnv.Location l = _env.makeLocation(17, 23);
            ASquareEnv.Location l1 = _env.makeLocation(17, 22);
            ASquareEnv.Location l2 = _env.makeLocation(18, 22);
            ASquareEnv.Location l3 = _env.makeLocation(18, 23);
            ASquareEnv.Location l4 = _env.makeLocation(17, 23);
            l = l.getNeighbor(_env.makeDirection(0, -1));
            assertEquals(true, l.same(l1));
            l = l.getNeighbor(_env.makeDirection(1, 0));
            assertEquals(true, l.same(l2));
            l = l.getNeighbor(_env.makeDirection(0, 1));
            assertEquals(true, l.same(l3));
            l = l.getNeighbor(_env.makeDirection(-1, 0));
            assertEquals(true, l.same(l4));

            // test wrapping
            ASquareEnv.Location w = _env.makeLocation(0, 0);
            ASquareEnv.Location w1 = _env.makeLocation(0, 49);
            ASquareEnv.Location w2 = _env.makeLocation(49, 49);
            ASquareEnv.Location w3 = _env.makeLocation(49, 0);
            ASquareEnv.Location w4 = _env.makeLocation(0, 0);
            w = w.getNeighbor(_env.makeDirection(0, -1));
            assertEquals(true, w.same(w1));
            w = w.getNeighbor(_env.makeDirection(-1, 0));
            assertEquals(true, w.same(w2));
            w = w.getNeighbor(_env.makeDirection(0, 1));
            assertEquals(true, w.same(w3));
            w = w.getNeighbor(_env.makeDirection(1, 0));
            assertEquals(true, w.same(w4));
        }
    }
}
