#include "read_pref_queue.h"

#define nil 0

#define WIFLAG 0x1 /* writer interested flag */
#define WAFLAG 0x2 /* writer active flag */
#define RDRCOUNT_INCR    0x4 /* increment for reader count */


RPQ_lock_init(L)
RPQlock *L;
{
	L->reader_head = nil;
	L->writer_tail = nil;
	L->writer_head = nil;
	L->rdr_cnt_and_flags = 0;
}

char *
RPQ_printname()
{
	return "RPQ";
}


#define INITIATE_WAKEUP_OF_BLOCKED_READERS(L) \
	{ \
		RPQqnode *head = (RPQqnode *) xmemul(&L->reader_head, nil); \
		if (head != nil) head->blocked = False; \
	}

#define CONDITIONALLY_RESUME_NEXT_WRITER(L) \
	{ \
		if (atomcas32(&L->rdr_cnt_and_flags, WIFLAG, WAFLAG) == WIFLAG) \
			L->writer_head->blocked = False; \
	}

void
RPQ_start_read(L, I)
RPQlock *L;
RPQqnode *I;
{
	if (atomadd32(&L->rdr_cnt_and_flags, RDRCOUNT_INCR) &  WAFLAG) {
		I->blocked = True;
		I->next = (RPQqnode *) xmemul(&L->reader_head, I);
		if ((L->rdr_cnt_and_flags & WAFLAG) == 0)
			INITIATE_WAKEUP_OF_BLOCKED_READERS(L);
		while (I->blocked != False);
		if (I->next != nil) I->next->blocked = False;
	}
}


void
RPQ_start_write(L, I)
RPQlock *L;
RPQqnode *I;
{
	RPQqnode *pred;
	if (atomcas32(&L->rdr_cnt_and_flags, 0, WAFLAG) == 0) {
		I->queued = False;
		return;
	}
	I->queued = True;
	I->blocked = True;
	I->next = nil;
	pred = (RPQqnode *) xmemul(&L->writer_tail, I);
	if (pred == nil) {
		L->writer_head = I;
		if (atomior32(&L->rdr_cnt_and_flags, WIFLAG) == 0) {
			/* worst case: the lock was freed between the initial cas and the 
			   ior so we need another cas */
			if (atomcas32(&L->rdr_cnt_and_flags, WIFLAG, WAFLAG) == WIFLAG) 
				return;
		}
	}
	else pred->next = I;
	while (I->blocked != False);
}


void
RPQ_end_read(L, I)
RPQlock *L;
RPQqnode *I;
{
	if (atomadd32(&L->rdr_cnt_and_flags, -RDRCOUNT_INCR) == 
			(WIFLAG + RDRCOUNT_INCR))
		CONDITIONALLY_RESUME_NEXT_WRITER(L);
}
	

void
RPQ_end_write(L, I)
RPQlock *L;
RPQqnode *I;
{
	/* clear WAFLAG and examine current state */ 
	unsigned long state = atomand32(&L->rdr_cnt_and_flags, ~WAFLAG); 

	if (I->queued == False) {
		/* if I wake up blocked readers, the last one to finish 
		   will wake up a queued writer since the writer at the head of the
		   queue will have set WIFLAG */
		if (state >= RDRCOUNT_INCR) { INITIATE_WAKEUP_OF_BLOCKED_READERS(L); }
		else if (state & WIFLAG) { CONDITIONALLY_RESUME_NEXT_WRITER(L); }
	}
	else {
		if (state >= RDRCOUNT_INCR) { INITIATE_WAKEUP_OF_BLOCKED_READERS(L); }
		/* I was queued, thus WIFLAG is clear and if a writer is queued 
		   behind me, I may have to wake him */
		if ((I->next != nil) || 
				(((RPQqnode *) atomcas32(&L->writer_tail, I, nil)) != I)) {
			while (I->next == nil);
			L->writer_head = I->next;
			if (atomior32(&L->rdr_cnt_and_flags, WIFLAG) == 0) {
				CONDITIONALLY_RESUME_NEXT_WRITER(L);
			}
		}
	}
}
