//===- OSR.cpp - Code to perform operator strength reduction --------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.  (Current version can be found
// at: http://llvm.org/releases/2.9/LICENSE.TXT)
//
//===----------------------------------------------------------------------===//
//
// This pass implements
// 
// Keith D. Cooper , L. Taylor Simpson , Christopher A. Vick, Operator strength 
// reduction, ACM Transactions on Programming Languages and Systems (TOPLAS), 
// v.23 n.5, p.603-625, September 2001.
// 
// "OSR improves upon an earlier algorithm of Allen, Cocke, and Kennedy 
// [Allen et al. 1981]. OSR operates on the static single assignment (SSA) form 
// of a procedure [Cytron et al. 1991]. By taking advantage of the properties 
// of SSA form, we have derived an algorithm that is simple to understand,
// quick to implement, and, in practice, fast to run. Its asymptotic complexity
// is,  in the worst case, the same as the Allen, Cocke,and Kennedy
// algorithm (ACK). 
// OSR achieves optimization results that are equivalent to those obtained 
// with the ACK algorithm."
//
//===----------------------------------------------------------------------===//
// A short explanation of the design of this code can be found on the
// Connexions website at: http://cnx.org/content/m41471/latest/
// 
//===----------------------------------------------------------------------===//
//This software was produced with support from the Defense Advanced
//Research Projects Agency (DARPA) through AFRL Contract FA8650-09-C-1915. 
//Nothing in this work should be construed as reflecting the official policy or
//position of the Defense Department, the United States government, or
//Rice University.
//
//THIS SOFTWARE HAS BEEN APPROVED FOR PUBLIC RELEASE, UNLIMITED DISTRIBUTION.
//
//===----------------------------------------------------------------------===//

#define DEBUG_TYPE "osr"

#include "llvm/Transforms/Scalar.h"
#include "llvm/Pass.h"
#include "llvm/Function.h"
#include "llvm/BasicBlock.h"
#include "llvm/Instructions.h"
#include "llvm/IntrinsicInst.h"
#include "llvm/Constants.h"
#include "llvm/DerivedTypes.h"
#include "llvm/Analysis/ConstantFolding.h"
#include "llvm/Analysis/CallGraph.h"
#include "llvm/Analysis/Dominators.h"
#include "llvm/Analysis/InstructionSimplify.h"
#include "llvm/Support/CFG.h"
#include "llvm/Support/Debug.h"
#include "llvm/Support/InstIterator.h"
#include "llvm/Support/GetElementPtrTypeIterator.h"
#include "llvm/ADT/DenseMap.h"
#include "llvm/ADT/SCCIterator.h"
#include "llvm/ADT/StringMap.h"
#include "llvm/ADT/Statistic.h"
#include "llvm/Transforms/Utils/BasicBlockUtils.h"
#include "llvm/Target/TargetData.h"

#include "llvm/Support/raw_ostream.h"

#include <map>
#include <vector>
#include <list>

using namespace llvm;

STATISTIC(NumOperatorsReduced, 
          "Number of operators reduced");
STATISTIC(NumInductionVariableAdded, 
          "Number of induction variables added");
STATISTIC(NumInductionVariableFound, 
          "Total number of induction variables");
STATISTIC(NumInductionVariableRemoved, 
          "Total number of induction variables removed "
          "via linear function test replacement");

STATISTIC(NumIteratorsTrashed, "Number of iterators thrashed");

namespace {
    
    class MultiKey {
      public:
        const Value *key1;
        const unsigned int    key2;
        Type  *key3;
        const Value *key4;
        
        MultiKey(const Value *k1, const int k2, Type *k3, const Value *k4)
          : key1(k1), key2(k2), key3(k3), key4(k4) { }
        
        bool operator<(const MultiKey &right) const 
        {
            if ( key1 == right.key1 )
                if ( key2 == right.key2 )
                    if ( key3 == right.key3 )
                        return key4 < right.key4;
                    else return key3 < right.key3;
                else return key2 < right.key2;
            else return key1 < right.key1;
        }    
    };

    class Transform {
      public:
        unsigned int opcode;
        Type *optype;
        Value *RC;
        Transform(Type *optype) 
          : opcode(0), optype(optype), RC(0) {};
        Transform(const Transform &Copy) 
          : opcode(Copy.opcode), optype(Copy.optype), RC(Copy.RC) {};
    };

    class InductionVariable : public Transform {
      public:
        int SCCId;
        bool noSignedWrap;
        bool noUnsignedWrap;
        int rponum;
        // for cleanup if OSR makes it useless
        Instruction *ext;
        unsigned int score;
        bool doneWithLFTR;

        InductionVariable *parent;
        std::list<InductionVariable*> children;

        InductionVariable(Type *optype) 
          : Transform(optype), SCCId(-1), noSignedWrap(false), 
             noUnsignedWrap(false), rponum(0), ext(0), score(0), 
             doneWithLFTR(false), parent(0) {
              Transform::opcode = 0;
              Transform::RC = 0;
        };

        InductionVariable(const InductionVariable &Copy) 
          : Transform(Copy), SCCId(Copy.SCCId), 
            noSignedWrap(Copy.noSignedWrap), 
            noUnsignedWrap(Copy.noUnsignedWrap), 
            rponum(Copy.rponum), ext(0), score(0), parent(0) {
        }    
        
        void addChild(InductionVariable *child) {
            std::list<InductionVariable*>::iterator i;
            InductionVariable *iv;
            
            for (i=children.begin(); i!=children.end(); ++i) {
                iv = *i;
                if ( iv->SCCId == child->SCCId ) {
                    return;  // do not want to add it twice
                }
            }
            
            children.push_back(child);
            child->parent = this;
        };
    };
    bool CompareIVs(const InductionVariable *IV1, 
                    const InductionVariable *IV2) {
        if ( IV1->rponum > IV2->rponum ) 
            return true;
        else {
            return false;
        }
    }
    
    
    class SCC {
      public:
        SCC() : headphi(0) { Insts.empty(); };
        std::vector<Instruction*> Insts;
        PHINode *headphi;
    };

    class SCCsT {
      private:
        std::vector<SCC> SCCs;

      public:
        int CreateNewSCC() {
            int SCCId;
            SCC EmptySCC;
            SCCs.push_back(EmptySCC);
            SCCId = SCCs.size() - 1;
            return SCCId;
        };
        
        void Pop() {
            SCCs.pop_back();
        };
        
        void Clear() {
            SCCs.clear();
        };

        void AddInstToSCC(int SCCId, Instruction *inst) {
            SCCs[SCCId].Insts.push_back(inst);
        };
        
        std::vector<Instruction*> GetSCC(int SCCId) {
            return SCCs[SCCId].Insts;
        }
        
        void SetHeadPhi(int SCCId, PHINode *phi) {
            if ( SCCId >= 0 ) {
                SCCs[SCCId].headphi = phi;
            }
            else {
                SCCs[SCCId].headphi = 0;
            }

        };
        
        PHINode *GetHeadPhi(int SCCId) {
            return ( SCCId == -1 ? 0 : SCCs[SCCId].headphi );
        };
        
        bool IsInstInSCC(int SCCId, Instruction *inst) {
            unsigned int j;
            bool found;
            found = false;
            for (j=0; j< SCCs[SCCId].Insts.size(); j++) {
                if ( SCCs[SCCId].Insts[j] == inst ) {
                    found = true;
                    break;
                }
            }
            return found;
        };
        
        int FindSCC(PHINode *headphi) {
            unsigned int i;
            for (i=0; i<SCCs.size(); i++) {
                if ( SCCs[i].headphi == headphi ) {
                    return int(i);
                }
            }
            return -1;
        };
        
        int FindSCC(Instruction *inst) {
            unsigned int i, j;
            for (i=0; i<SCCs.size(); i++) {
                for (j=0; j< SCCs[i].Insts.size(); j++) {
                    if ( SCCs[i].Insts[j] == inst ) {
                        return i;
                    }
                }
            }
            return -1;
        };

        void ReplaceInst(Instruction *oldinst, Instruction *newinst) {
            unsigned int i, j;
            for (i=0; i<SCCs.size(); i++) {
                for (j=0; j< SCCs[i].Insts.size(); j++) {
                    if ( SCCs[i].Insts[j] == oldinst ) {
                        SCCs[i].Insts[j] = newinst;
                    }
                }
            }
        };

        void RemoveInst(Instruction *inst) {
            unsigned int i, j;
            for (i=0; i<SCCs.size(); i++) {
                for (j=0; j< SCCs[i].Insts.size(); ) {
                    if ( SCCs[i].Insts[j] == inst ) {
                        SCCs[i].Insts.erase(SCCs[i].Insts.begin()+j);
                    }
                    else {
                        j++;
                    }
                }
            }
        };
        
        void Dump() {
            unsigned int i, j;
            for (i=0; i<SCCs.size(); i++) {
                dbgs() << "SCC#" << i << ":\n";
                for (j=0; j< SCCs[i].Insts.size(); j++) {
                    if ( SCCs[i].Insts[j] == SCCs[i].headphi ) {
                        dbgs() << "\t<<<" << *SCCs[i].Insts[j] << ">>>\n";
                    }
                    else {
                        dbgs() << "\t" << *SCCs[i].Insts[j] << "\n";
                    }
                }
            }
        };
    };
    
    
    struct OSR : public FunctionPass {
    public:
        static char ID; // Pass identification, replacement for typeid

        OSR() : FunctionPass(ID) {};
        
        virtual bool runOnFunction(Function& F);
        
        virtual void getAnalysisUsage(AnalysisUsage& AU) const {
            AU.addRequired<DominatorTree>();
            AU.setPreservesCFG();
        }
        
    private:
        //===--------------------------------------------------------------===//
        // Functions defined in the paper.
        //===--------------------------------------------------------------===//
        
        void runOSR(Function& F);
        
        bool DFS(Instruction *I);
        
        bool ProcessSCC(std::vector<Instruction*> SCC);
        
        bool ClassifyIV(std::vector<Instruction*> SCC, int SCCId);
        
        bool RegionConst(const Value *V, const BasicBlock *LoopHeader);
        
        void Replace(Instruction *I, Instruction **IV, Value **RC,
                     std::list<InductionVariable*> Transforms,
                     int OldSCCId, bool *updateInstructions);

        Instruction* Reduce(Instruction **IV, 
                            unsigned int opcode, Type *optype, 
                            Value **RC, 
                            std::list<InductionVariable*> Transforms,
                            int OldSCCId, int NewSSCId,
                            bool *updateInstructions);
        
        Value* Apply(Value **op1, 
                     unsigned int opcode, Type *optype, 
                     Value **op2, 
                     std::list<InductionVariable*> Transforms,
                     BasicBlock *header = 0);
        
        //===--------------------------------------------------------------===//
        // Functions that could have been defined in the paper.
        //===--------------------------------------------------------------===//
        
        Value* ApplyNoReduce(Value *op1, 
                             unsigned int opcode, Type *optype, 
                             Value *op2, 
                             std::list<InductionVariable*> Transforms);
        
        Value* Convert(Value *op1, 
                       unsigned int opcode, Type *optype, 
                       BasicBlock *header, Instruction *insertBefore);
        
        bool IsIdentityTransfrom(Value *op1, 
                                 unsigned int opcode, Type *optype, 
                                 Value *op2, Value **Identity = 0);
        
        Value *Extend(Value *op, 
                      std::list<InductionVariable*> Transforms,
                      BasicBlock *header = 0);

        Value *ExtendConstant(ConstantInt *op, 
                              std::list<InductionVariable*> Transforms);
        
        Value *ExtendInstruction(Value *op, 
                                 std::list<InductionVariable*> Transforms,
                                 BasicBlock *header = 0);
        
        //===--------------------------------------------------------------===//
        // Linear Function Test Replacement
        //===--------------------------------------------------------------===//

        SCCsT SCCs;

        void ScoreReplacement(InductionVariable *IV);

        void FindReplacements(InductionVariable *IV,
                              std::list<InductionVariable*> *Replacements);

        void LFTR(Function& F);
        
        //===--------------------------------------------------------------===//
        // Support data and functions.
        //===--------------------------------------------------------------===//
        
        void DumpIR(Function& F);
        void DumpNodes();
        void DumpIVs(std::list<InductionVariable*> IVs, int indent = 0);

        llvm::DenseMap<Instruction*, bool>          Visited;
        llvm::DenseMap<Instruction*, int>           Low;
        llvm::DenseMap<Instruction*, int>           DFSnum;
        llvm::DenseMap<Instruction*, BasicBlock*>   Header;
        llvm::DenseMap<BasicBlock*, bool>           BBVisited;
        llvm::DenseMap<BasicBlock*, int>            RPOnum;
        
        std::list<InductionVariable*> InductionVariables;
        InductionVariable *FindIV(std::list<InductionVariable*> IVs,
                                  int  SSCId);
        void AddIV(InductionVariable *oldiv, 
                   std::list<InductionVariable*> *Transforms,
                   InductionVariable **newiv);
        void AddIV(InductionVariable **IV);
        void FreeIVs(std::list<InductionVariable*> *IVs);
        
        const TargetData *TD;
        unsigned PtrSizeInBits;

        DominatorTree *DT;
        
        int nextDFSnum;
        
        Instruction *FindApplyInsertLocation(Value *op1, Value *op2);
        
        std::vector<Instruction*>  Nodes;
        
        bool IsOperandOnStack(Instruction *o) {
            return ( find(Nodes.begin(), Nodes.end(), o) != Nodes.end() );
        }
        
        bool IsInductionVariable(Value *op, 
                                 Instruction **iv,
                                 std::list<InductionVariable*> *Transforms);

        struct Cracked { 
            unsigned int opcode; 
            bool isOp1IV; 
            bool foundCandidate;
            Instruction *IV; 
            Value *RC; 
            std::list<InductionVariable*> transforms;
            Cracked(): opcode(0), isOp1IV(false), foundCandidate(false), 
                       IV(0), RC(0) {}; 
        };
        
        Cracked CrackInstruction(Instruction *I);
        
        void RPO(BasicBlock *BB);
        
        void ReplaceInstruction(Instruction *OldInst, Instruction *NewInst);
        
        void ReplaceInstructionUse(Instruction *OldInst, Instruction *NewInst);
        
        void ReplaceInductionVariable(Instruction *OldInst, 
                                      Instruction *NewInst);
        
        void DeleteInstruction(Instruction *OldInst);

        std::map<MultiKey, Instruction*> AddedInstructions;
        
        //===--------------------------------------------------------------===//
        // BuildHashKey() creates a string from the hash key,
        // for debug only purposes
        //===--------------------------------------------------------------===//
        
        void BuildHashKey(const Value *IV, const unsigned int opcode, 
                          Type *optype, const Value *RC, 
                          char *key, const int keysize) {
            snprintf(key, keysize, "%#.16lx %#.8x %#.16lx %#.16lx",
                     (const unsigned long)IV, 
                     opcode, (const unsigned long)optype, 
                     (const unsigned long)RC);
        }
        
        Instruction* SearchHashTable(const Value *IV, 
                                     const unsigned int opcode, 
                                     Type *optype,
                                     const Value *RC);
        
        void AddHashTable(const Value *IV, 
                          const unsigned int opcode,  
                          Type *optype,
                          const Value *RC, 
                          Instruction *inst);
        
        void UpdateHashTable(Value *oldValue, 
                             Value *newValue);
        
        void DumpHashTable(const char *label);
        
        PHINode* SelectPHI(Instruction *phi1, Instruction *phi2) {
            // pick the phi with the lower RPONum
            // ie the phi in the SCC's header
            int rponum1 = RPOnum[phi1->getParent()];
            int rponum2 = RPOnum[phi2->getParent()];
            if ( rponum1 < rponum2 ) {
                return dyn_cast<PHINode>(phi1);
            }
            else if ( rponum2 < rponum1 ) {
                return dyn_cast<PHINode>(phi2);
            }
            else {
                // got a tie, SCC contains at least two PHI nodes
                // in the same header basic block.
                // use pointer comparison to break
                if ( phi1 < phi2 ) {
                    return dyn_cast<PHINode>(phi1);
                }
                else {
                    return dyn_cast<PHINode>(phi2);
                }
            }
        };

        void SetNoWrap(Instruction *I, Instruction *iv);
    };
    
    // LLVM uses the address of OSR::ID 
    // as the pass's unque id (and not the actual value thereof)
    char OSR::ID = 0;
    
}


INITIALIZE_PASS(
    OSR                           /* C++ class name */,
    "osr"                         /* Command line switch for optimization */, 
    "Operator Strength Reduction" /* Command line description */, 
    false                         /* Only looks at CFG */,
    false                         /* Analysis Pass */
)


//===----------------------------------------------------------------------===//
// defined in llvm/Transforms/Scalar.h
//
// called from dynmically unreachable code in ForcePassLinking::ForcePassLinking 
// in llvm/LinkAllPasses.h.  the above construct is needed to force this .cpp 
// file to be linked and not treated as "dead code".
//
// this is required but but not covered by the Writing an LLVM Pass tutorial.
//===----------------------------------------------------------------------===//

FunctionPass *llvm::createOSRPass() {
    assert("this code should never be called.");
    return new OSR();
}


//===----------------------------------------------------------------------===//
///  runOnFunction - driver method for Operator Strength Reduction
///  optimization pass.
//===----------------------------------------------------------------------===//

bool OSR::runOnFunction(Function& F) {
    // target data is specified in the LLVM IR:
    //     target datalayout = "layout specification"
    TD = getAnalysisIfAvailable<TargetData>();
    PtrSizeInBits = TD->getPointerSizeInBits();

    DT = &getAnalysis<DominatorTree>();
    
    DEBUG(dbgs() << "\n\nVisit function: " << F.getName() << "\n";);
    DEBUG(DumpIR(F););
    
    //
    // Calculate the RPO numbering for the basic blocks.
    //
    BBVisited.clear();
    RPOnum.clear();
    SCCs.Clear();
    nextDFSnum = F.size();
    RPO(F.begin());
    
    //
    // Perform Operator Strength Reduction for Function F.
    //
    runOSR(F);

    DEBUG(dbgs() << "\n\nDone with OSR for " << F.getName() << "\n\n";);
    DEBUG(DumpIR(F););
    DEBUG(DumpIVs(InductionVariables););
    
    //
    // Perform Linear Function Test Replacement for Function F.
    //
    LFTR(F);
    DEBUG(DumpIR(F););
    
    return true;
}


//===----------------------------------------------------------------------===//
/// OSR - performs the Operator Strength Reduction optimization, as described in
/// the following paper:
/// Keith D. Cooper , L. Taylor Simpson , Christopher A. Vick, Operator strength
/// reduction, ACM Transactions on Programming Languages and Systems (TOPLAS), 
/// v.23 n.5, p.603-625, September 2001.
//===----------------------------------------------------------------------===//

void OSR::runOSR(Function& F) {
/*
 OSR(SSAgraph) 
     while there is an unvisited node n in SSAgraph
         DFS(n) 
 */
    bool visitedInstruction;
    bool updatedInstructions;
    
    Visited.clear();
    Low.clear();
    DFSnum.clear();
    Header.clear();
    AddedInstructions.clear();
    InductionVariables.clear();
    nextDFSnum = 0;

    do {
        visitedInstruction = false;
        Function::iterator BBI = F.begin();
        Function::iterator BBE = F.end();
        for (;  BBI != BBE;  ++BBI) 
        {
            //BasicBlock &BB = *BBI;
            BasicBlock::iterator II = BBI->begin();
            BasicBlock::iterator IE = BBI->end();
            while (II != IE) 
            {
                Instruction *I = II; // iterator is intrinsically a pointer
                if ( I == 0 ) {
                    // Instruction Iterator needs to be reset
                    NumIteratorsTrashed++;
                    II = BBI->begin();
                    IE = BBI->end();
                    continue;
                }

                updatedInstructions = false;
                if ( Visited[I] ) {
                    DEBUG(dbgs() << "already visited:" << *I << "\n";);
                }
                else {
                    DEBUG(dbgs() << "not visited:" << *I << "\n";);
                    updatedInstructions = DFS(I);
                    //visitedInstruction = true;
                }
                if ( updatedInstructions ) {
                    // ++II can have undefined behavour if its Instruction
                    // has been replaced via reduction.  Thus, we restart.
                    II = BBI->begin();
                    IE  = BBI->end();
                }
                else {
                    ++II;
                }
            }
            //if ( visitedInstruction ) break;
        }
    } while (visitedInstruction);
}

//===----------------------------------------------------------------------===//
/// DumpIR - Dump the Function's IR (debug only)
//===----------------------------------------------------------------------===//

void OSR::DumpIR(Function& F) {
#if 1
    F.dump();
#else
    dbgs() << "\n"
           <<  "function: " << F.getName() << "\n";
    for (Function::iterator BBI = F.begin(), E = F.end();  BBI != E; 
         ++BBI) {
        dbgs() << "\n" << BBI->getName() << ":\n";
        for (BasicBlock::iterator II = BBI->begin(), E = BBI->end(); 
             II != E; ++II) {
            Instruction *I = II;
            char hexI[1024];
            snprintf(hexI, 1024, "%#.16lx", (unsigned long)I);
            dbgs() << hexI << ":" << *I << "\n";
            if ( isa<BranchInst>(I) ) continue;
            int i;
            int NumOperands;
            Value *op;
            NumOperands = I->getNumOperands();
            for (i=0; i<NumOperands; i++) {
                op = I->getOperand(i);
                snprintf(hexI, 1024, "%#.16lx", (unsigned long)op);
                dbgs() << "\t#" << i << " : " <<  hexI << " : ";
                if ( isa<BasicBlock>(op) ) {
                    dbgs() << "\n";
                }
                else {
                    dbgs() << *op << "\n";
                }
            }
        }
    }
    dbgs() << "\n";
#endif
}

void OSR::DumpNodes() {
    int size, i;
    Instruction *I;
    
    size = Nodes.size();
    for (i=0; i<size; i++) {
        I = Nodes[i];
        char hexI[1024];
        snprintf(hexI, 1024, "%#.16lx", (unsigned long)I);
        dbgs() << "Nodes[" << i << "] = " << hexI << " = " << *I << "\n";
    }
}


void OSR::DumpIVs(std::list<InductionVariable*> IVs, int indent) {
    char indent_str[1024];
    std::list<InductionVariable*>::iterator i;
    InductionVariable *iv;
    Instruction *phi;
    bool isExtension;
    int SCCId;
    
    memset(indent_str, ' ', sizeof(indent_str));
    indent_str[indent] = '\0';
    
    for (i=IVs.begin(); i!=IVs.end(); ++i) {
        iv = *i;
        isExtension = ( iv->opcode == Instruction::SExt || 
                        iv->opcode == Instruction::ZExt );
        SCCId = iv->SCCId;
        phi = SCCs.GetHeadPhi(SCCId);
        if ( phi != 0 ) {
            dbgs() << indent_str << "SCC Id#:" << SCCId << "\n"
                   << indent_str << "RPO#: " << iv->rponum << "\n"
                   << indent_str << "Induction variable head phi: " << *phi 
                   << "\n";
        } else {
            dbgs() << indent_str << "Induction variable: nada\n";
        }
        // Instruction::getOpcodeName(opcode)
        dbgs() << indent_str << "Opcode: " 
               << Instruction::getOpcodeName(iv->opcode) << "\n";
        dbgs() << indent_str << "Type: " << *iv->optype << "\n";
        if ( !isExtension ) {
            dbgs() << indent_str << "RC : ";
            if ( iv-> RC != 0 ) {
                dbgs() << *iv->RC << "\n";
            }
            else {
                dbgs() << "nada\n";
            }
        }
        if ( !isExtension ) {
            dbgs() << indent_str << "Arithemtic overflow " 
                   << ( iv->noSignedWrap ? "does not" : "does" ) 
                   << " signed wrap.\n";
            dbgs() << indent_str << "Arithemtic overflow " 
                   << ( iv->noUnsignedWrap ? "does not" : "does" ) 
                   << " unsigned wrap.\n";
        }
        dbgs() << indent_str << "Has " << iv->children.size() << " children.\n";
        dbgs() << "\n";
        if ( iv->children.size() > 0 ) {
            DumpIVs(iv->children, indent+4);
        }
    }
}


//===----------------------------------------------------------------------===//
/// FindIV - look to see if I is in the IVs list, recursively.
//===----------------------------------------------------------------------===//

InductionVariable *OSR::FindIV(std::list<InductionVariable*> IVs,
                               int SCCId) {
    std::list<InductionVariable*>::iterator i;
    InductionVariable *iv;
    
    for (i=IVs.begin(); i!=IVs.end(); ++i) {
        iv = *i;
        if ( iv->SCCId == SCCId ) {
            return iv;
        } else if ( iv->children.size() > 0 ) {
            iv = FindIV(iv->children, SCCId);
            if ( iv ) {
                return iv;
            }
        }
    }
    
    return 0;
}
 

//===----------------------------------------------------------------------===//
/// AddIV - add a 'found' induction variable to global InductionVariables.
/// Input is freed if it can not be added.
//===----------------------------------------------------------------------===//

void OSR::AddIV(InductionVariable **IV) {
    InductionVariable *existingiv;

    existingiv = FindIV(InductionVariables, (*IV)->SCCId);
    if ( existingiv == 0 ) {
        InductionVariables.push_back(*IV);
    } else {
        delete *IV;
    }
    *IV = 0;
}


//===----------------------------------------------------------------------===//
/// AddIV - add a created induction variable to global InductionVariables.
/// Transforms paramter has logial induction variable(s) which need their
/// own entries in InductionVariables.
//===----------------------------------------------------------------------===//

void OSR::AddIV(InductionVariable *oldiv, 
                std::list<InductionVariable*> *Transforms,
                InductionVariable **newiv) {
    InductionVariable *currentiv;
    std::list<InductionVariable*>::iterator itrans;
    std::list<InductionVariable*>::iterator ichild;
    InductionVariable *child;
    InductionVariable *extiv;
    InductionVariable *copyiv;
    std::list<InductionVariable*> children;
    bool bFound;    
    
    // oldiv is a node in the InductionVariables tree
    currentiv = oldiv;
    
    //
    // add Transforms (list of sext/zext instructions) 
    // to global InductionVariables.
    //

    for (itrans=Transforms->begin(); itrans!=Transforms->end(); ++itrans) {
        Transform *transform = *itrans;
        bFound = false;
        children = currentiv->children;
        for (ichild=children.begin(); ichild!=children.end(); ++ichild) {
             child = *ichild;
            if ( transform->opcode == child->opcode &&
                 transform->optype == child->optype &&
                 transform->RC     == child->RC ) {
                currentiv = child;
                bFound = true;
                break;
            }
        }
        if ( !bFound ) {
            extiv = static_cast<InductionVariable*>(transform);
            copyiv = new InductionVariable(*extiv);
            currentiv->addChild(copyiv);
            currentiv = copyiv;
            copyiv = 0;
        }
    }
    
    //
    // add newiv to global InductionVariables.
    //
    
    bFound = false;
    children = currentiv->children;
    for (ichild=children.begin(); ichild!=children.end(); ++ichild) {
        child = *ichild;
        if ( (*newiv)->opcode == child->opcode &&
             (*newiv)->optype == child->optype &&
             (*newiv)->RC     == child->RC ) {
            bFound = true;
            break;
        }
    }
    if ( !bFound ) {
        copyiv = new InductionVariable(**newiv);
        // hackish: maybe newiv should have this set instead?
        copyiv->noSignedWrap = oldiv->noSignedWrap;
        copyiv->noUnsignedWrap = oldiv->noUnsignedWrap;
        currentiv->addChild(copyiv);
        copyiv = 0;
    } 
}


void OSR::FreeIVs(std::list<InductionVariable*> *IVs) {
    std::list<InductionVariable*>::iterator iviter;
    InductionVariable *iv;
    for (iviter = IVs->begin(); iviter != IVs->end(); ++iviter) {
        iv = *iviter;
        if ( iv->children.size() > 0 ) {
            FreeIVs(&iv->children);
        }
        delete iv; iv = 0;
    }
    IVs->clear();
}


//===----------------------------------------------------------------------===//
/// DFS - does a DFS of the SSA graph for this function. Strongly Connected
/// Components (SCC) that are discovered  with more than one node will be 
/// candidates for induction variables.
//===----------------------------------------------------------------------===//

bool OSR::DFS(Instruction *I) {
/*
 DFS(node)
     node.DFSnum ← nextDFSnum++ 
     node.visited ← TRUE 
     node.low ← node.DFSnum 
     PUSH(node) 
     for each o ∈ {operands of node}
       if not o.visited 
         DFS(o)
         node.low ← MIN(node.low, o.low)
       if o.DFSnum < node.DFSnum and o ∈ stack
           node.low ← MIN(o.DFSnum, node.low) 
     if node.low = node.DFSnum
       SCC ← ∅ 
       do
         x ← POP()
         SCC ← SCC ∪ {x} 
       while x ̸= node 
       ProcessSCC(SCC)
 */
    Instruction *o;
    bool updatedInstructions;
    bool update;
    bool recheckVisited;
    int i, NumOperands;
    
    DEBUG(char hexI[1024];
          snprintf(hexI, 1024, "%#.16lx", (unsigned long)I);
          dbgs() << "DFS(" << I->getParent()->getName() << ":" << hexI << ":";
          dbgs() << *I << ")\n";);
    
    
    updatedInstructions = false;
    
    DFSnum[I] = nextDFSnum++;
    Visited[I] = true;
    Low[I] = DFSnum[I];
    Nodes.push_back(I);
    
    NumOperands = I->getNumOperands();
    
    do {
        // instructions on the Nodes stack can have their operands
        // changed during strength reduction and need to be rechecked.
        recheckVisited = false;

        for (i=0; i<NumOperands; i++) {
            Value *operand = I->getOperand(i);
            if ( operand == 0 ) continue;
            if ( isa<Instruction>(*operand) ) {
                o = dyn_cast<Instruction>(operand);
                if ( !Visited[o] ) {
                    update = DFS(o);
                    updatedInstructions |= update;
                    recheckVisited |= update;
                    Low[I] = ( Low[o] < Low[I] ? Low[o] : Low[I] );
                }
                if ( DFSnum[o] < DFSnum[I] and IsOperandOnStack(o) ) {
                    Low[I] = ( DFSnum[o] < Low[I] ? DFSnum[o] : Low[I] );
                }
            }
        }
    } while ( recheckVisited );

    if ( Low[I] == DFSnum[I] ) {
        Instruction *x;
        std::vector<Instruction*> SCC;
        do {
            x = Nodes.back();
            Nodes.pop_back();
            SCC.push_back(x);
        } while( x != I );
        
        DEBUG(if ( SCC.size() > 1 ) dbgs() << "found a loop.\n";);
        
        update = ProcessSCC(SCC);
        updatedInstructions |= update;
    }
    
    return updatedInstructions;
}


//===----------------------------------------------------------------------===//
/// ProcessSCC - will discover induction variables and attempt to strenght
/// reduce instructions.
//===----------------------------------------------------------------------===//

bool OSR::ProcessSCC(std::vector<Instruction*> SCC) {
/*
 ProcessSCC(SCC)
     if SCC has a single member n
         if n is of the form x ← iv × rc, x ← rc × iv, 
                             x ← iv ± rc, or x ← rc + iv 
             Replace(n, iv, rc)
         else
             n.header ← NULL
     else
         ClassifyIV(SCC)
 */
    Instruction *I;
    PHINode *phi;
    bool updatedInstructions;
    int SCCId;
    Instruction *thisinst;
    
    updatedInstructions = false;
    if ( SCC.size() == 1 ) {
        I = SCC[0];
        DEBUG(
            char hexI[1024];
            snprintf(hexI, 1024, "%#.16lx", (unsigned long)I);
            dbgs() << "SCC single node DFS #" << DFSnum[I] << " :" 
                   << hexI << ":" << *I << "\n";
        );
        Cracked cracked = CrackInstruction(I);
        if ( cracked.foundCandidate  ) {
            SCCId = SCCs.FindSCC(cracked.IV);
            Replace(I, &cracked.IV, &cracked.RC, cracked.transforms, 
                    SCCId, &updatedInstructions);
            FreeIVs(&cracked.transforms);
        }
        else {
            Header[I] = 0; // only IVs have Header set
        }
    } else {
        phi = 0;
        // sleuth out the head phi in the SCC
        for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
            I != E;  ++I)
        {
            thisinst = *I;
            if ( thisinst->getOpcode() == Instruction::PHI ) {
                // there can be more than one PHI per SCC
                if ( phi == 0 ) {
                    phi = dyn_cast<PHINode>(thisinst);
                }
                else {
                    // pick the phi with the higher RPO Numbering
                    // ie the phi in the SCC's header
                    phi = SelectPHI(phi, thisinst);
                }
            }
        }
        // see if SCCs entry has already been made for this SCC
        // if not, make a new SCCs entry
        SCCId = SCCs.FindSCC(phi);
        if ( SCCId == -1 ) {
            SCCId = SCCs.CreateNewSCC();
            for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
                I != E;  ++I)
            {
                thisinst = *I;
                SCCs.AddInstToSCC(SCCId, thisinst);
            }
            SCCs.SetHeadPhi(SCCId, phi);
            DEBUG(SCCs.Dump(););
        }
        
        updatedInstructions = ClassifyIV(SCC, SCCId);
    }
    return updatedInstructions;
}


//===----------------------------------------------------------------------===//
/// ClassifyIV - will determine if the SCC loop is an induction variable.  
/// If not, attempt to strenght reduce the instructions.
//===----------------------------------------------------------------------===//

bool OSR::ClassifyIV(std::vector<Instruction*> SCC, int SCCId) {
/*
 ClassifyIV(SCC)
     for each n ∈ SCC
         if header→RPOnum > n.block→RPOnum 
             header ← n.block
     for each n ∈ SCC 
         if n.op not ∈ {φ, +, −, COPY}
             SCC is not an induction variable 
         else
             for each o ∈ {operands of n} 
                 if o not ∈ SCC and not RegionConst(o, header)
                     SCC is not an induction variable 
     if SCC is an induction variable
         for each n ∈ SCC
             n.header ← header
     else 
         for each n ∈ SCC
             if n is of the form x ← iv × rc, x ← rc × iv, 
                                 x ← iv ± rc, or x ← rc + iv 
                 Replace(n, iv, rc)
             else
                 n.header ← NULL
 */
    BasicBlock *header;
    Instruction *thisinst;
    PHINode *phi;
    unsigned int opcode;
    bool isInductionVariable;
    int NumOperands;
    bool isOperandInSCC;
    bool isOperandRegionConstant;
    bool isOp1InSCC;
    bool isOp1RegionConstant;
    bool isOp2InSCC;
    bool isOp2RegionConstant;
    Value *operand;
    Value *op1;
    Value *op2;
    int incr;
    bool noSignedWrap;
    bool noUnsignedWrap;
    unsigned int sizeInBits; // PtrSizeInBits
    bool updateInstructions;
    int OldSCCId;
    
    updateInstructions = false;
    
    isInductionVariable = true;
    
    thisinst = *SCC.begin();
    header = thisinst->getParent();
    
    phi = SCCs.GetHeadPhi(SCCId);
    
    noSignedWrap = false;
    noUnsignedWrap = false;
    sizeInBits = 0;
    for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
        I != E;  ++I)
    {
        thisinst = *I;
        DEBUG(
            char hexI[1024];
            snprintf(hexI, 1024, "%#.16lx", (unsigned long)thisinst);
            dbgs() << "SCC loop node DFS #" << DFSnum[thisinst] << ":" 
                   << hexI << ":" << *thisinst << "\n";
        );

        if ( RPOnum[header] > RPOnum[thisinst->getParent()] ) {
            header = thisinst->getParent();
        }

        opcode = thisinst->getOpcode();
        if ( opcode == Instruction::Add || opcode == Instruction::Sub ) {
            BinaryOperator *binop;
            binop = dyn_cast<BinaryOperator>(thisinst);
            if ( binop->hasNoSignedWrap() ) {
                noSignedWrap = true;
            }
            if ( binop->hasNoUnsignedWrap() ) {
                noUnsignedWrap = true;
            }
        }
    }
    
    Type *optype = phi->getType();
    if ( optype->isIntegerTy() ) {
        sizeInBits = cast<IntegerType>(optype)->getBitWidth();
    }
    else if ( optype->isPointerTy() ) {
        sizeInBits = PtrSizeInBits;
    }
    
    //
    // determine if this SCC represents an induction variable
    //

    for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
        I != E;  ++I)
    {
        thisinst = *I;
        opcode = thisinst->getOpcode();
        // Include the casting ops here? Algorithm calls for searching
        // COPY instructions here, but LLVM IR does not have a COPY (since
        // a COPY is just an alias on exisitng DEF).  LLVM IR does have type
        // cast ops (SExt, ZExt, BitCast) but they change the type (doooooh).
        // IV ops must have the same type, since theirs associated PHI operands
        // must have the same types.
        if ( opcode == Instruction::Add || opcode == Instruction::Sub ||
             opcode == Instruction::PHI ) 
        {
            NumOperands = thisinst->getNumOperands();
            incr = ( opcode == Instruction::PHI ? 2 : 1 );
            for (int i=0; i<NumOperands; i+=incr) 
            {
                operand = thisinst->getOperand(i);
                if ( isa<UndefValue>(operand) ) {
                    // operand is undefined aka undef
                    isInductionVariable = false;
                    break;
                }
                isOperandInSCC =  
                  ( find(SCC.begin(), SCC.end(), operand) != SCC.end() );
                isOperandRegionConstant = RegionConst(operand, header);
                if ( ! isOperandInSCC and ! isOperandRegionConstant ) {
                    isInductionVariable = false;
                    isOperandRegionConstant = RegionConst(operand, header);
                    break;
                }
            }
            // fixes significant oversight of the original algorithm 
            // from paper.
            if ( opcode == Instruction::Add || opcode == Instruction::Sub )
            {
                op1 = thisinst->getOperand(0);
                op2 = thisinst->getOperand(1);
                isOp1InSCC =  
                  ( find(SCC.begin(), SCC.end(), op1) != SCC.end() );
                isOp2InSCC =  
                  ( find(SCC.begin(), SCC.end(), op2) != SCC.end() );
                if ( isOp1InSCC && isOp2InSCC ) {
                    // i = i + i  algbraicly the same as 
                    // i = 2 * i which is not allowed
                    isInductionVariable = false;
                }
            }
        }
        else if ( opcode == Instruction::GetElementPtr ) {
            // GEP RC, SCCnode is like RC + SCCnode
            // FIXME: only checking the 2 operand form?
            op1 = thisinst->getOperand(0);
            if ( isa<UndefValue>(op1) ) {
                // operand is undefined aka undef
                isInductionVariable = false;
                break;
            }
            op2 = thisinst->getOperand(1);
            if ( isa<UndefValue>(op2) ) {
                // operand is undefined aka undef
                isInductionVariable = false;
                break;
            }
            isOp1InSCC =  
              ( find(SCC.begin(), SCC.end(), op1) != SCC.end() );
            isOp1RegionConstant = RegionConst(op1, header);
            isOp2InSCC =  
              ( find(SCC.begin(), SCC.end(), op2) != SCC.end() );
            isOp2RegionConstant = RegionConst(op2, header);
            if ( isOp1InSCC && isOp2RegionConstant ) {
                // op1 is in SCC and op2 is a RC
            }
            else if ( isOp2InSCC && isOp1RegionConstant ) {
                // op2 is in SCC and op1 is a RC
            }
            else {
                isInductionVariable = false;
                break;
            }
        }
#if 0
        else if ( opcode == Instruction::BitCast ) {
            operand = thisinst->getOperand(0);
            if ( isa<UndefValue>(operand) ) {
                // operand is undefined aka undef
                isInductionVariable = false;
                break;
            }
            isOperandInSCC =  
              ( find(SCC.begin(), SCC.end(), operand) != SCC.end() );
            if ( !isOperandInSCC ) {
                isInductionVariable = false;
                break;
            }
        }
#endif
        else {
            isInductionVariable = false;
            break;
        }
    }
    
    //
    // apply a second filter here to the SCC.  original OSR paper assumed
    // that address and "int" integers had the same size.  thus, adding one
    // to an int and to an address would have the same overflow properties.
    // when they have different sizes like having 32 bit int and a 64 bit 
    // address, an add that would overflow an integer would not overflow
    // an address.  thus in this case, promoting an add by one from an int 
    // to an address behave differently in the presence of overflow.
    //
    // LLVM has 'no wrap' attributes for its integer adds and subtracts.
    // the attribute encapsulates the semantics for the C language.  In C,
    // signed ints do not overflow (while unsigned do).  On oveflow, signed 
    // ints have an undtermined value and the program enters in an unknown
    // state.  other places in LLVM take advantage of this by assuming that
    // the program is never in an unknown state.  for example, expressions like
    // "int(i) < int(i+1)" evaluate to always "true", which may or may not be 
    // the case if i+1 overflows.
    //
    // thus, OSR can be applied to 'no wrap' ints which are a smaller size
    // than an address.  the overflow of int induction variable will not
    // have a similar overflow in the corresponding address calculation.
    // this divergence is acceptable since the program enters an unknown 
    // state.
    //
    
    if ( isInductionVariable ) {
        if ( sizeInBits > PtrSizeInBits ) {
            // unlikely but ...
            isInductionVariable = false;
        }
        else if ( sizeInBits < PtrSizeInBits ) {
            if ( !noSignedWrap && !noUnsignedWrap ) {
                isInductionVariable = false;
            }
        }
        else {
            // IV candidate and address have the same size,
            // or the candidate is smaller and does not wrap on overflow.
        }

    }

    if ( isInductionVariable ) {
        DEBUG(dbgs() << "found induction variable.\n";);
        InductionVariable *newIV;
        Instruction *first;
        NumInductionVariableFound++;
        
        first = *SCC.begin();
        newIV = new InductionVariable(first->getType());
        for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
            I != E;  ++I)
        {
            thisinst = *I;
            Header[thisinst] = header;
        }
        newIV->SCCId = SCCId;
        newIV->noSignedWrap = noSignedWrap;
        newIV->noUnsignedWrap = noUnsignedWrap;
        newIV->rponum = RPOnum[phi->getParent()];
        
        AddIV(&newIV);
        // side effect: newIV = 0;
    }
    else {
        DEBUG(dbgs() << "did not find induction variable.\n";);

        // we create some IVs which ClassifyIV() does not recognize,
        // so we need unmark them as IVs. CrackInstruction() uses Header[]
        // to determine IV and misbehaves if ClassifyIV() and Header[]
        // are not in agreement.
        for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
            I != E;  ++I)
        {
            thisinst = *I;
            Header[thisinst] = 0;
        }

        for(std::vector<Instruction*>::iterator I = SCC.begin(), E = SCC.end();
            I != E;  ++I)
        {
            thisinst = *I;

            Cracked cracked = CrackInstruction(thisinst);
            if ( cracked.foundCandidate ) {
                OldSCCId = SCCs.FindSCC(cracked.IV);
                Replace(thisinst, &cracked.IV, &cracked.RC, cracked.transforms,
                        OldSCCId, &updateInstructions);
                FreeIVs(&cracked.transforms);
            }
            else {
                Header[thisinst] = 0;
            }

        }
    }
    
    return updateInstructions;
}


//===----------------------------------------------------------------------===//
/// RegionConst - determine if the given value is a constant or has a constant 
/// value over the loop body, which is defined by the loop header basic block.
//===----------------------------------------------------------------------===//

bool OSR::RegionConst(const Value *V, 
                      const BasicBlock *LoopHeader) {
/*
 RegionConst(name, header)
     return name.op = LOAD IMMEDIATE or name.block ≫ header
 */
    if ( V == 0 ) {
        return false;
    }
    else if ( isa<Constant>(V) ) {
        return true;
    }
    else if ( isa<Argument>(V) ) {
        return true;
    }
    else if ( isa<Instruction>(V)) {
        if ( LoopHeader == 0 ) return  false;
        
        const Instruction *I;
        const BasicBlock *DefinitionBB;
        bool isSameBasicBlock;
        
        I = dyn_cast<Instruction>(V);
        DefinitionBB = I->getParent();
        isSameBasicBlock = ( DefinitionBB == LoopHeader );
        // DefinitionBB dominates Loop Header => region constant
        if ( !isSameBasicBlock ) {
            if ( DT->dominates(DefinitionBB, LoopHeader) ) {
                return true;
            }
        }
    }
    return false;
}

//===----------------------------------------------------------------------===//
/// Replace - transform Instruction I into a new induction variable.
//===----------------------------------------------------------------------===//

void OSR::Replace(Instruction *I, Instruction **IV, Value **RC,
                  std::list<InductionVariable*> Transforms,
                  int OldSCCId, bool *updateInstructions) {
/*
 Replace(node, iv, rc)
     result ← Reduce(node.op, iv, rc)
     Replace node with a COPY from result 
     node.header ← iv.header
 */
    Instruction *result;
    unsigned int opcode;
    Type *optype;
    InductionVariable *oldiv;
    InductionVariable *newiv;
    int NewSCCId;
    
    // initialize the output parameters
    *updateInstructions = false;

    DEBUG(dbgs() << "Replace(\n\tI=" << *I << ",\n"
                 << "\tIV=" << **IV << ",\n" 
                 << "\tRC=" << **RC << ")\n";);
    
    opcode = I->getOpcode();
    optype = I->getType();
    NewSCCId = SCCs.CreateNewSCC();
    result = Reduce(IV, opcode, optype, RC, Transforms, 
                    OldSCCId, NewSCCId, updateInstructions);
    if ( I != result ) {
        // Replace node with a COPY from result?  Not the LLVM way. 
        DEBUG(dbgs() << "replacing \n"
                     << "\t" << *I << "\n"
                     << "with\n"
                     << "\t" << *result << "\n";
        );
                
        oldiv = FindIV(InductionVariables, OldSCCId);

        ReplaceInstructionUse(I, result);
        NumOperatorsReduced++;

        // above replace, may orphan some zext/sext instruction in Transforms
        // find and remove any of those orphans
        std::list<InductionVariable*>::reverse_iterator riteriv;
        for(riteriv=Transforms.rbegin(); riteriv!=Transforms.rend(); ++riteriv)
        {
            InductionVariable *iv;
            iv = *riteriv;
            if ( iv->ext->use_empty() ) {
                DeleteInstruction(iv->ext);
                iv->ext = 0;
            } else {
                break;
            }
        }

        if ( *updateInstructions ) {
            DEBUG(dbgs() << "new phi is\n"
                         << "\t" << *SCCs.GetHeadPhi(NewSCCId) << "\n";);
            DEBUG(SCCs.Dump(););

            NumInductionVariableAdded++;
            
            newiv = new InductionVariable(result->getType());
            newiv->SCCId = NewSCCId;
            newiv->opcode = opcode;
            newiv->RC = *RC;
            newiv->rponum = RPOnum[SCCs.GetHeadPhi(NewSCCId)->getParent()];

            AddIV(oldiv, &Transforms, &newiv);
            delete newiv; newiv = 0;
            
            DEBUG(DumpIVs(InductionVariables););
        }
        else {
            // did not create a new SCC and did not populate the SCC instance 
            // that we created before the call to Reduce(), so we pop the SCCs stack.
            SCCs.Pop();
        }
        
        // ReplaceInstructionUse() updates the instructions stream
        *updateInstructions |= true;
                
        DEBUG(Function *F= result->getParent()->getParent();
              DumpIR(*F););
    } else {
        DEBUG(dbgs() << "not replacing:\n"
                     << "\t" << *I << "\n"
                     << "it is already reduced.\n";);
        SCCs.Pop();
    }
    
    return;
}


//===----------------------------------------------------------------------===//
/// Reduce - Inserts code to strength reduce an induction variable and returns 
/// the newly created induction variable.
//===----------------------------------------------------------------------===//

Instruction* OSR::Reduce(Instruction **IV, 
                         unsigned int opcode, 
                         Type *optype,
                         Value **RC,
                         std::list<InductionVariable*> Transforms,
                         int OldSCCId, int NewSCCId,
                         bool *updateInstructions) {
/*
 SSAname Reduce(opcode, iv, rc)
     result ← search(opcode, iv, rc)
     if result is not found 
         result ← inventName() 
         add(opcode, iv, rc, result) 
         newDef ← copyDef(iv, result) 
         newDef.header ← iv.header 
         for each operand o of newDef
             if o.header = iv.header
                 Replace o with Reduce(opcode, o, rc)
             else if opcode = mult or newDef.op = φ
                 Replace o with Apply(opcode, o, rc)
     return result
 */
    // Create a new induction variable with the same shape as the original, 
    // but possibly with a different initial value or a different increment.
    Instruction *newDef;
    Instruction *oinst;
    Instruction *result;
    Value *o;
    Value *newo;
    Value *convertedo;
    BasicBlock *bb;
    bool hasAlreadyBeenAdded;
    int i;
    int NumOperands;
    int incr;
    
    result = SearchHashTable(*IV, opcode, optype, *RC);
    hasAlreadyBeenAdded = ( result != 0 );
    DEBUG(dbgs() << "Reduce(\n"
                 << "\tIV=" << **IV << ",\n"
                 << "\topcode=" << Instruction::getOpcodeName(opcode) << ",\n"
                 << "\tconst optype=" << *optype << ",\n"
                 << "\tconst RC=" << **RC << ") "
                 << ( hasAlreadyBeenAdded 
                        ? "has already been reduced\n" 
                        : "has not been reduced\n" ););
    if ( !hasAlreadyBeenAdded ) {        
        // Instruction is an abstract class ...
        //Instruction *newerDef = new Instruction(IV->getOpcode(), optype, 
        //                                        IV->OperandList, 
        //                                        IV->getNumOperands());
        // clone calls virtual clone_impl() and then adds back metadata
        // optype is implicitly set? can not be explicitly set?
        
        //
        // Check for algebraic identity reduction (like adding zero)
        // and early exit in that case.
        //
        if ( Transforms.empty() ) {
            bool isIdentity;
            
            isIdentity = IsIdentityTransfrom(*IV, opcode, optype, *RC);
            if ( isIdentity ) {
                // Reduce() has nothing to do.
                result = *IV;
                *updateInstructions |= false;

                newDef = *IV;
                AddHashTable(*IV, opcode, optype, *RC, newDef);
                
                SCCs.AddInstToSCC(NewSCCId, result);

                DEBUG(dbgs() << "Reduce(\n"
                             << "\tIV=" << **IV << ",\n"
                             << "\topcode=" << Instruction::getOpcodeName(opcode) << ",\n"
                             << "\tconst optype=" << *optype << ",\n"
                             << "\tconst RC=" << **RC << ")\n"
                             << "returns : " << *result << "\n";);
                
                return result;
            }
        }
    
        // clone parameter IV, new instruction may have a different optype
        // param IV is an induction variable and is limited to
        // phi, +, - and * opertors, see ClassifyIV()
        // note that not all SCC loops will be classified as IV.
        
        if ( opcode == Instruction::GetElementPtr &&
             (*IV)->getOpcode() == Instruction::Add ) {
            // IV0 takes the form IV0 <- IV + RC or RC + IV
            // need to create a GEP IV, RC
            Value *op1;
            op1 = (*IV)->getOperand(0);
            if ( RegionConst(op1, Header[*IV]) ) {
                // IV0 <- RC + IV and need GEP IV, RC
                // => swap operands on IV0
                BinaryOperator *binop;
                binop = dyn_cast<BinaryOperator>(*IV);
                binop->swapOperands();
            }
            Constant *zero;    // place holder
            Constant *nullptr; // place holder
            zero = Constant::getNullValue((*IV)->getType());
            nullptr = Constant::getNullValue(optype);
            newDef = GetElementPtrInst::Create(nullptr, zero, 
                                               Twine(), *IV);
        } else if ( opcode == Instruction::GetElementPtr &&
                    (*IV)->getOpcode() == Instruction::Sub ) {
            // newIV <- IV - RC
            // need to create GEP IV, -RC
            Value *op1;
            Value *op2;
            Value *negateop;
            BinaryOperator *newaddop;
            Constant *zero;    // place holder
            Constant *nullptr; // place holder
            Instruction *negateinst;
            // need to create -RC <- 0 - RC and 
            // to create replacementIV <- IV + -RC
            op1 = (*IV)->getOperand(0);
            op2 = (*IV)->getOperand(1);
            if ( isa<Instruction>(*op2) || isa<Argument>(*op2) ) {
                // negate op2 via calculating 0 - op2
                // not 100% safe
                negateop = BinaryOperator::CreateNeg(op2);

                zero = Constant::getNullValue(op1->getType());
                negateinst = dyn_cast<Instruction>(negateop);
                if ( isa<Instruction>(*op2) ) {
                    negateinst->insertAfter(dyn_cast<Instruction>(op2));
                }
                else if ( isa<Argument>(*op2) ) {
                    negateinst->insertBefore(DT->getRoot()->getTerminator());
                }
                newaddop = BinaryOperator::Create(Instruction::Add, 
                                                  op1, negateop, 
                                                  Twine(), *IV);
                SetNoWrap(newaddop, *IV);
            }
            else if ( isa<ConstantInt>(*op2) ) {
                ConstantInt *constint = dyn_cast<ConstantInt>(op2);
                APInt IntConst(constint->getValue());
                Constant *NegatedConst;
                NegatedConst = Constant::getIntegerValue(constint->getType(),
                                                         -constint->getValue());
                
                newaddop = BinaryOperator::Create(Instruction::Add, 
                                                  op1, NegatedConst, 
                                                  Twine(), *IV);
                SetNoWrap(newaddop, *IV);
                
            }
            // create GEP replacementIV, -RC
            zero = Constant::getNullValue((*IV)->getType());
            nullptr = Constant::getNullValue(optype);
            newDef = GetElementPtrInst::Create(nullptr, zero, 
                                               Twine(), *IV);                                       
                                                               
            // replace newIV <- IV - RC with replacementIV <- IV + -RC
            DEBUG(dbgs() << "nuking:\n\t" << **IV << "\n";);
            ReplaceInductionVariable(*IV, newaddop);
            *IV = newaddop;
            DEBUG(dbgs() << "IV is now:\n\t" << **IV << "\n";);
        } else if ( (*IV)->getOpcode() == Instruction::GetElementPtr &&
                    opcode == Instruction::BitCast ) {
            // converting from one pointer to another and
            // can not reduce this everytime
            BitCastInst *bitcast;
            bitcast = new BitCastInst(*IV, optype, Twine());
            bitcast->insertAfter(*IV);
            newDef = bitcast;
            AddHashTable(*IV, opcode, optype, *RC, newDef);
            Header[newDef] = 0;
            result = newDef;
            *updateInstructions |= true;
            SCCs.AddInstToSCC(NewSCCId, result);
            DEBUG(dbgs() << "Reduce(\n"
                   << "\tIV=" << *IV << ",\n"
                   << "\topcode=" << Instruction::getOpcodeName(opcode) << ",\n"
                   << "\tconst optype=" << *optype << ",\n"
                   << "\tconst RC=" << *RC << ") returns : " 
                   << *result << "\n";);
            return result;
#if 0
        } else if ( IV->getOpcode() == Instruction::BitCast &&
                    opcode == Instruction::BitCast ) {
            BitCastInst *bitcast;
            Constant *zero;    // place holder
            zero = Constant::getNullValue(optype);
            bitcast = new BitCastInst(zero, IV->getType(), Twine(), IV);
            newDef = bitcast;
#endif
       } else if ( (*IV)->isBinaryOp() ) {
            BinaryOperator *binop;
            Instruction::BinaryOps binopcode;
            Constant *zero; // place holder
            zero = Constant::getNullValue(optype);
            binopcode = Instruction::BinaryOps((*IV)->getOpcode());
            binop = BinaryOperator::Create(binopcode, 
                                           zero, zero, 
                                           Twine(), 
                                           *IV);
            SetNoWrap(binop, *IV);
            newDef = binop;
        } else if ( (*IV)->getOpcode() == Instruction::PHI ) {
            PHINode *phinode;
            PHINode *sccheadphi;
            NumOperands = (*IV)->getNumOperands();
            phinode = PHINode::Create(optype, NumOperands/2, Twine(), *IV);
            newDef = phinode;
            sccheadphi = SCCs.GetHeadPhi(NewSCCId);
            if ( sccheadphi == 0 ) {
                SCCs.SetHeadPhi(NewSCCId, phinode);
            }
            else {
                SCCs.SetHeadPhi(NewSCCId, 
                                SelectPHI(sccheadphi, phinode));
            }

        } else if ( (*IV)->getOpcode() == Instruction::GetElementPtr ) {
            Constant *zero;    // place holder
            Constant *nullptr; // place holder
            zero = Constant::getNullValue(Type::getInt64Ty((*IV)->getContext()));
            nullptr = Constant::getNullValue(optype);
            newDef = GetElementPtrInst::Create(nullptr, zero, 
                                               Twine(), *IV);
        } else if ( optype == (*IV)->getType() ) {
            newDef = (*IV)->clone();
            newDef->insertAfter(*IV);
        } else {
            assert(false && "I had hope that it would never come to this.");
            newDef = 0;
        }

        AddHashTable(*IV, opcode, optype, *RC, newDef);
        Header[newDef] = Header[*IV];
        result = newDef;

        // "reduce" operands
        NumOperands = (*IV)->getNumOperands();
        incr = ( (*IV)->getOpcode() == Instruction::PHI ? 2 : 1 );
        for (i=0; i<NumOperands; i+=incr) {
            o = (*IV)->getOperand(i);
            if ( isa<Instruction>(*o) && 
                 Header[dyn_cast<Instruction>(o)] == Header[*IV] &&
                 SCCs.IsInstInSCC(OldSCCId, dyn_cast<Instruction>(o)) ) {
                oinst = dyn_cast<Instruction>(o);
                newo = Reduce(&oinst, opcode, optype, RC, Transforms,
                              OldSCCId, NewSCCId, updateInstructions);
                if ( newDef->getOpcode() == Instruction::PHI ) {
                    PHINode *phinode = dyn_cast<PHINode>(newDef);
                    if ( isa<Instruction>(newo) ) {
                        bb = dyn_cast<BasicBlock>((*IV)->getOperand(i+1));
                    } else {
                        bb = &((*IV)->getParent()->getParent()->getEntryBlock());
                        bb = dyn_cast<BasicBlock>((*IV)->getOperand(i+1));
                    }
                    phinode->addIncoming(newo, bb);
                } else {
                    newDef->setOperand(i, newo);
                }
            } else if ( isa<PHINode>(**IV) ) {
                PHINode *phinode = dyn_cast<PHINode>(newDef);
                Value *extendedo;
                extendedo = Extend(o, Transforms, Header[*IV]);
                newo = Apply(&extendedo, opcode, optype, RC, Transforms);
                if ( isa<Instruction>(newo) ) {
                    bb = dyn_cast<BasicBlock>((*IV)->getOperand(i+1));
                } else {
                    bb = &((*IV)->getParent()->getParent()->getEntryBlock());
                    bb = dyn_cast<BasicBlock>((*IV)->getOperand(i+1));
                }
                DEBUG(dbgs() << "adding to phi node:\n"
                             << "\t" << *phinode << "\n"
                             << "\twhich has the type: " << *phinode->getType() 
                             << "\n";
                      dbgs() << "adding:\n"
                             << "\t" << *newo << "\n"
                             << "\twhich has type: " << *newo->getType() 
                             << "\n";);
                phinode->addIncoming(newo, bb);
            } else if ( opcode == Instruction::Mul ) {
                Value *extendedo;
                extendedo = Extend(o, Transforms, Header[*IV]);
                newo = Apply(&extendedo, opcode, optype, RC, Transforms);
                newDef->setOperand(i, newo);
            } else if ( opcode == Instruction::GetElementPtr ) {
                // newIV = GEP newPHI, RC
                // the first operand PHI is handled above
                // thus we get here for the second operand RC
                // RC can be a constant or an instruction that has a constant
                // value for the loop. The former can be heavily optimized.
                if ( isa<Constant>(o) ) {
                    // the result of the implicit multiply in GEP
                    // is a compile time constant
                    newDef->setOperand(i, Extend(o, Transforms, Header[*IV]));
                } else {
                    // the result of the implicit multiply in GEP
                    // is a run time constant.  Need to convert
                    //   GEP newPHI, zero
                    // to
                    //   %1 = cast float* newPHI to char*
                    //   %2 = GEP char* %1, coverted(o)
                    //   %3 = cast %2 to float*
                    //   delete GEP newPHI, zero
                    // where float* is the optype
                    
                    // optype is char* => multiplying by one
                    // this can be heavily optimized like above
                    // maybe use getPrimitiveSizeInBits()
                    
                    PointerType *byteptrtype; 
                    Type *Int8ty;
                    unsigned AddressSpace;
                    BitCastInst *cast;
                    GetElementPtrInst *gep;
                    
                    Int8ty = Type::getInt8Ty((*IV)->getContext());
                    AddressSpace = 0; // FIXME: no idea if this is right
                    byteptrtype = PointerType::get(Int8ty, AddressSpace);
                    
                    if ( optype != byteptrtype ) {
                        //   %1 = cast float* newPHI to char*
                        cast = new BitCastInst(newDef->getOperand(0), 
                                               byteptrtype, Twine(), newDef);
                        //   %2 = GEP char* %1, coverted(o)
                        convertedo = Convert(Extend(o, Transforms, Header[*IV]),
                                             opcode, optype, Header[*IV], 
                                             newDef);
                        gep = GetElementPtrInst::Create(cast, convertedo, 
                                                        Twine(), newDef);
                        //   %3 = cast %2 to float*
                        cast = new BitCastInst(gep, optype, Twine(), newDef);
                        
                        //   delete GEP newPHI, zero
                        ReplaceInstruction(newDef, cast);
                        newDef = cast;
                    }
                    else {
                        //   %1 = GEP char* newPHI, coverted(o)
                        convertedo = Convert(Extend(o, Transforms, Header[*IV]),
                                             opcode, optype,Header[*IV], newDef);
                        gep = GetElementPtrInst::Create(newDef->getOperand(0), 
                                                        convertedo, Twine(), 
                                                        newDef);
                        //   delete GEP newPHI, zero
                        ReplaceInstruction(newDef, gep);
                        newDef = gep;
                    }

                    result = newDef;
                }
                // deal with GEP, see ConstantFoldLoadThroughGEPConstantExpr?
            } else if ( o->getType() != optype ) {
                Value *extendedo;
                extendedo = Extend(o, Transforms, Header[*IV]);
                newo = Apply(&extendedo, opcode, optype, RC, Transforms, Header[newDef]);
                newDef->setOperand(i, newo);
            } else {
                DEBUG(dbgs() << "NOT REPLACING OPERAND\n"
                             << "\t" << *o <<"\n"
                             << "in Reduce().\n";);
                newDef->setOperand(i, o);
            }
        }
        // made reductions => added instruction
        *updateInstructions |= true;

        SCCs.AddInstToSCC(NewSCCId, result);        
    }
    else {
        // this reduction has already been done
        *updateInstructions |= false;
    }
    
    DEBUG(dbgs() << "Reduce(\n"
                 << "\tIV=" << **IV << ",\n"
                 << "\topcode=" << Instruction::getOpcodeName(opcode) << ",\n"
                 << "\tconst optype=" << *optype << ",\n"
                 << "\tconst RC=" << **RC << ")\n"
                 << "returns : " << *result << "\n";);

    return result;
}


//===----------------------------------------------------------------------===//
/// Apply - Inserts an instruction to apply an opcode to two operands and 
/// returns the new instruction. The opcode + optype and second operand
/// is applied to the first operand.
//===----------------------------------------------------------------------===//

Value* OSR::Apply(Value **op1, 
                  unsigned int opcode, Type *optype,
                  Value **op2,
                  std::list<InductionVariable*> Transforms,
                  BasicBlock *header /* =0 */) {
/*
 SSAname Apply(opcode, op1, op2)
     result ← search(opcode, op1, op2)
     if result is not found
         if op1.header ̸= NULL and RegionConst(op2, op1.header) 
             result ← Reduce(opcode, op1, op2)
         else if op2.header ̸= NULL and RegionConst(op1, op2.header) 
             result ← Reduce(opcode, op2, op1)
         else 
             result ← inventName() 
             add(opcode, op1, op2, result) 
             Choose the location where the operation will be inserted 
               see findNearestCommonDominator()
               "the instruction must go into a block that is dominated by the 
               definitions of both operands"
             Decide if constant folding is possible 
             Create newOper at the desired location 
             newOper.header ← NULL
     return result
 */
    Value *result;
    bool hasAlreadyBeenAdded;
    bool updateInstructions;
    bool isOp1IV;
    bool isOp2IV;
    bool isOp1RC;
    bool isOp2RC;
    int OldSCCId;
    int NewSCCId;
    SCC EmptySCC;
    Instruction *IV;
    
    if ( isa<Instruction>(**op1) ){
        result = SearchHashTable(dyn_cast<Instruction>(*op1), 
                                 opcode, optype, 
                                 *op2);
    }
    else {
        result = 0;
    }
    
    hasAlreadyBeenAdded = ( result != 0 );
    DEBUG(dbgs() << "Apply(\n"
                 << "\top1=" << **op1 << ",\n"
                 << "\topcode=" << Instruction::getOpcodeName(opcode)  << ",\n"
                 << "\toptype=" << *optype  << ",\n"
                 << "\top2=" << **op2 << ") "
                 << ( hasAlreadyBeenAdded 
                        ? "has already been reduced\n" 
                        : "has not been reduced\n" ););
    
    if ( !hasAlreadyBeenAdded ) {
        isOp1IV = ( op1 && isa<Instruction>(**op1) &&
                    Header[cast<Instruction>(*op1)] != 0 );
        isOp2IV = ( op2 && isa<Instruction>(**op2) &&
                    Header[dyn_cast<Instruction>(*op2)] != 0 );
        isOp1RC = RegionConst(*op1, 
                              *op1 && isa<Instruction>(**op1) 
                                ? Header[cast<Instruction>(*op1)]
                                : 0ul);
        isOp2RC = RegionConst(*op2, 
                              *op2 && isa<Instruction>(**op2) 
                                ? Header[dyn_cast<Instruction>(*op2)]
                                : 0ul);
        if ( isOp1IV) {
            OldSCCId = SCCs.FindSCC(cast<Instruction>(*op1));
        } else if ( isOp2IV ) {
            OldSCCId = SCCs.FindSCC(cast<Instruction>(*op2));
        } else {
            OldSCCId = -1;
        }

        if ( isOp1IV && isOp2RC && OldSCCId > 0 ) {
            IV = cast<Instruction>(*op1);
            updateInstructions = false;
            NewSCCId = SCCs.CreateNewSCC();
            result = Reduce(&IV, opcode, optype, op2, 
                            Transforms, OldSCCId, NewSCCId,
                            &updateInstructions);
            DEBUG(SCCs.Dump(););
            if ( result != IV ) {
                if ( updateInstructions ) {
                    // Reduce() creates a new a new induction variable
                    InductionVariable *oldiv = FindIV(InductionVariables, OldSCCId);
                    InductionVariable *newiv = new InductionVariable(result->getType());
                    newiv->SCCId = NewSCCId;
                    newiv->opcode = opcode;
                    newiv->RC = *op2;
                    newiv->rponum = RPOnum[SCCs.GetHeadPhi(NewSCCId)->getParent()];
                    AddIV(oldiv, &Transforms, &newiv);
                    delete newiv; newiv = 0;
                    DEBUG(DumpIVs(InductionVariables););
                }
                else {
                    SCCs.Pop();
                }
            }
            else {
                SCCs.Pop();
            }
        }
        else if ( isOp2IV && isOp1RC && OldSCCId > 0 ) {
            IV = cast<Instruction>(*op2);
            updateInstructions = false;
            NewSCCId = SCCs.CreateNewSCC();
            result = Reduce(&IV, opcode, optype, op1, 
                            Transforms, OldSCCId, NewSCCId,
                            &updateInstructions);
            DEBUG(SCCs.Dump(););
            if ( result != IV ) {
                if ( updateInstructions ) {
                    // Reduce() creates a new a new induction variable
                    InductionVariable *oldiv = FindIV(InductionVariables, OldSCCId);
                    InductionVariable *newiv = new InductionVariable(result->getType());
                    newiv->SCCId = NewSCCId;
                    newiv->opcode = opcode;
                    newiv->RC = *op1;
                    newiv->rponum = RPOnum[SCCs.GetHeadPhi(NewSCCId)->getParent()];
                    AddIV(oldiv, &Transforms, &newiv);
                    delete newiv; newiv = 0;
                    DEBUG(DumpIVs(InductionVariables););
                }
                else {
                    SCCs.Pop();
                }
            }
            else {
                SCCs.Pop();
            }
        }
        else {
            result = ApplyNoReduce(*op1, opcode, optype, *op2, Transforms);
        }
    }
    
    DEBUG(dbgs() << "Apply(\n"
                 << "\top1=" << **op1 << ",\n"
                 << "\topcode=" << Instruction::getOpcodeName(opcode)  << ",\n"
                 << "\toptype=" << *optype  << ",\n"
                 << "\top2=" << **op2 << ")\n"
                 << "returns : " << *result << "\n";);

    return result;
}


//===----------------------------------------------------------------------===//
/// ApplyNoReduce - Inserts an instruction to apply an opcode to two operands 
/// and returns the new instruction. The opcode + optype and second operand
/// is applied to the first operand.  No new Induction Variables are created
/// via a call to Reduce.
//===----------------------------------------------------------------------===//

Value* OSR::ApplyNoReduce(Value *op1, 
                          unsigned int opcode, Type *optype,
                          Value *op2,
                          std::list<InductionVariable*> Transforms) {
    Value *result;
    bool hasAlreadyBeenAdded;
    Instruction *insertBefore;

    if ( isa<Instruction>(*op1) ){
        result = SearchHashTable(dyn_cast<Instruction>(op1), 
                                 opcode, optype, 
                                 op2);
    }
    else {
        result = 0;
    }
    
    hasAlreadyBeenAdded = ( result != 0 );
    if ( !hasAlreadyBeenAdded ) {
        //decide if constant folding is posiible
        if ( (op1 && isa<Constant>(op1)) && 
             (op2 && isa<Constant>(op2)) ) {
            Constant *folded;
            SmallVector<Constant *, 2> Operands;
            Operands.push_back(dyn_cast<Constant>(op1));
            Operands.push_back(dyn_cast<Constant>(op2));
            if ( opcode == Instruction::GetElementPtr ) {
                // ConstantFoldInstOperands() requires the first operand
                // to be the pointer for the GEP opcode.
                if ( op2->getType()->isPointerTy() ) {
                    Operands[0] = dyn_cast<Constant>(op2);
                    Operands[1] = dyn_cast<Constant>(op1);
                }
            }
            folded = ConstantFoldInstOperands(opcode, op1->getType(),
                                              Operands);
            if ( folded->getType() == optype ) {
                result = folded;
            } 
            else {
                bool isConstantInteger;
                ConstantInt *folded_int;
                
                folded_int = dyn_cast<ConstantInt>(folded);
                isConstantInteger = ( folded_int != 0 );
                if ( isConstantInteger ) {
                    int32_t BitWidth;
                    ConstantInt *casted;
                    
                    BitWidth = cast<IntegerType>(optype)->getBitWidth();
                    APInt Result(folded_int->getValue());
                    // FIXME: see APSInt& extend(uint32_t width)
                    Result.sext(BitWidth);
                    casted = ConstantInt::get(folded_int->getContext(), 
                                              Result);
                    result = casted;
                }
            }
        } 
#if 0
        else if ( op1 && isa<Constant>(op1) && op2 == 0 ) {
            if ( op1->getType() != optype ) {
                ConstantInt *consop_int;
                bool isConstantInteger;
                consop_int = dyn_cast<ConstantInt>(op1);
                isConstantInteger = ( consop_int != 0 );
                if ( isConstantInteger ) {
                    int32_t BitWidth;
                    ConstantInt *casted;
                    
                    BitWidth = cast<IntegerType>(optype)->getBitWidth();
                    APInt Result(consop_int->getValue());
                    Result.sext(BitWidth);
                    casted = ConstantInt::get(consop_int->getContext(), 
                                              Result);
                    result = casted;
                }
            } else if ( opcode == Instruction::Sub ) {
                // compute -op1 (aka negation) via 0 - op1
                ConstantInt *consop_int;
                Constant *zero;
                Constant *folded;
                consop_int = dyn_cast<ConstantInt>(op1);
                zero = Constant::getNullValue(optype);
                Constant *Operands[2] = { zero, consop_int };
                folded = ConstantFoldInstOperands(opcode, op1->getType(),
                                                  Operands, 2);
                result = folded;
            }
        }
#endif
        if ( result == 0 ) {
            if ( Instruction::isBinaryOp(opcode) ) {
                // create a binary instruction
                BinaryOperator *binop;
                Instruction::BinaryOps binopcode;
                
                Value *simplify;
                binopcode = Instruction::BinaryOps(opcode);
                simplify = SimplifyBinOp(binopcode, op1, op2);
                if ( simplify != 0 ) {
                    result = simplify;
                }
                // SimplifyBinOp() punts on multiplication 
                else if ( opcode == Instruction::Mul && op1 && 
                         isa<ConstantInt>(op1) && 
                         dyn_cast<ConstantInt>(op1)->isZero() ) {
                    result = op1;
                }
                else if ( opcode == Instruction::Mul && op2 && 
                         isa<ConstantInt>(op2) && 
                         dyn_cast<ConstantInt>(op2)->isZero() ) {
                    result = op2;
                }
                else if ( opcode == Instruction::Mul && op1 && 
                         isa<ConstantInt>(op1) && 
                         dyn_cast<ConstantInt>(op1)->isOne() ) {
                    result = op2;
                }
                else if ( opcode == Instruction::Mul && op2 && 
                         isa<ConstantInt>(op2) && 
                         dyn_cast<ConstantInt>(op2)->isOne() ) {
                    result = op1;
                }
                else {
                    insertBefore = FindApplyInsertLocation(op1, op2);
                    binop = BinaryOperator::Create(binopcode, op1, op2, 
                                                   Twine(), 
                                                   insertBefore);
                    result = binop;
                    AddHashTable(op1, opcode, optype, op2, binop);
                }
            }
            // opcode can not be SExt/ZExt when called from OSR
            // but can be when called from LFTR.  OSR calls Extend()
            // instead.
            else if ( opcode == Instruction::SExt ) {
                SExtInst *sextinst;
                
                insertBefore = FindApplyInsertLocation(op1, op2);
                sextinst = new SExtInst(op1, optype, Twine(), insertBefore);
                AddHashTable(op1, opcode, optype, op2, sextinst);
                result = sextinst;
            }
            else if ( opcode == Instruction::ZExt ) {
                ZExtInst *zextinst;
                
                insertBefore = FindApplyInsertLocation(op1, op2);
                zextinst = new ZExtInst(op1, optype, Twine(), insertBefore);
                AddHashTable(op1, opcode, optype, op2, zextinst);
                result = zextinst;
            }
            else if ( opcode == Instruction::BitCast ) {
                BitCastInst *bitcast;
                
                insertBefore = FindApplyInsertLocation(op1, op2);
                bitcast = new BitCastInst(op1, optype, Twine(), insertBefore);
                AddHashTable(op1, opcode, optype, op2, bitcast);
                result = bitcast;
            }
            else if ( opcode == Instruction::GetElementPtr ) {
                GetElementPtrInst *gep;
                bool needExtraGepOperand;
                Value *address;
                Value *index;
                
                if ( op1->getType()->isPointerTy() ) {
                    address = op1;
                    index = op2;
                } else {
                    address = op2;
                    index = op1;
                }
                
                needExtraGepOperand = false;
                
                Type *addresstype = address->getType();
                
                DEBUG(dbgs() << "addresstype is " << *addresstype << "\n";
                      dbgs() << "target optype is " << *optype << "\n";);
                if ( addresstype != optype ) 
                {
                    // this happns when we need to add RC to
                    // [16 x float]* and need to get float*
                    needExtraGepOperand = true;
                }
                
                insertBefore = FindApplyInsertLocation(op1, op2);
                if ( needExtraGepOperand ) {
                    SmallVector<Value*, 16> Indices;
                    Constant *zero;
                    Type *i64type = Type::getInt64Ty(op2->getContext());
                    zero = Constant::getNullValue(i64type);
                    Indices.push_back(zero);
                    Indices.push_back(index);
                    gep = GetElementPtrInst::Create(address, Indices, 
                                                    Twine(), insertBefore);
                }
                else {
                    gep = GetElementPtrInst::Create(address, index, Twine(),
                                                    insertBefore);
                }
                if ( gep->getType() == optype ) {
                    AddHashTable(op1, opcode, optype, op2, gep);
                    result = gep;
                } else {
                    // cast our troubles away
                    BitCastInst *bitcast;
                    bitcast = new BitCastInst(gep, optype, Twine());
                    bitcast->insertAfter(gep);
                    AddHashTable(op1, opcode, optype, op2, bitcast);
                    result = bitcast;
                }
            }
        }
    }
    
    return result;
}


//===----------------------------------------------------------------------===//
/// Convert - Inserts an instruction to convert operands via the pair 
/// opcode + optype.
//===----------------------------------------------------------------------===//

Value* OSR::Convert(Value *op1, 
                    unsigned int opcode, 
                    Type *optype,
                    BasicBlock *header,
                    Instruction *insertBefore) {
    /*
     SSAname Convert(opcode, op1, op2)
         result ← search(opcode, op1, op2)
         if result is not found
             Choose the location where the operation will be inserted 
             convert op1 to new type optype via opcode
             add(opcode, op1, null, result) 
     return result
     */
    Value *result;
    Instruction *resultinst;
    bool hasAlreadyBeenAdded;
    bool isOp1RC;
    std::list<InductionVariable*>::iterator iteriv;
    
    result = SearchHashTable(op1, opcode, optype, 0);
    
    hasAlreadyBeenAdded = ( result != 0 );
    DEBUG(dbgs() << "Convert(\n"
                 << "\top1=" << *op1 << ",\n"
                 << "\topcode=" << Instruction::getOpcodeName(opcode)  << ",\n"
                 << "\toptype=" << *optype  << ") "
                 << ( hasAlreadyBeenAdded 
                        ? "has already been reduced\n" 
                        : "has not been reduced\n" ););
    
    if ( !hasAlreadyBeenAdded ) {
        isOp1RC = RegionConst(op1, header);

        if ( opcode == Instruction::GetElementPtr ) {
            
            if ( isOp1RC ) {
                // if op1 is a Region Constant, it can be defined as a function
                // argument or as a variable outside.  first we need to find a
                // Basic Block where the converted defintion can live.
                if ( isa<Argument>(op1) ) {
                    // definition of op1 is as an argument
                    Argument *arg;
                    DomTreeNode *dtnode;
                    DomTreeNode *idom;
                    arg = dyn_cast<Argument>(op1);
                    dtnode = DT->getNode(header);
                    idom = ( dtnode ? dtnode->getIDom() : 0 );
                    if ( idom ) {
                        insertBefore = idom->getBlock()->getTerminator();
                    }
                    else {
                        BasicBlock *entryBB;
                        entryBB = &(arg->getParent()->getEntryBlock());
                        insertBefore = entryBB->getTerminator();
                    }
                }
                else if ( isa<Instruction>(op1) ) {
                    Instruction *inst;
                    inst = dyn_cast<Instruction>(op1);
                    insertBefore = inst->getParent()->getTerminator();
                }
            } else {
                // insertBefore = insertBefore;
            }
            
            // operating on a GEP addr(op1), int(RC) 

            if ( op1->getType()->isIntegerTy() ) {
                Value *gep;
                Instruction *castinst;
                Constant *nullptr;
                // create GEP null, RC(index), 
                // performs the implicit multiplication
                nullptr = Constant::getNullValue(optype);
                gep = GetElementPtrInst::Create(nullptr, op1, Twine(),
                                                insertBefore);
                // covert pointer back to int
                castinst = new PtrToIntInst(gep, op1->getType(), Twine(), 
                                            insertBefore);
                resultinst = castinst;
                AddHashTable(op1, opcode, optype, 0, resultinst);
                result = resultinst;
            }
            else if ( op1->getType()->isPointerTy() ) {
                if ( op1->getType() != optype ) {
                    BitCastInst *bitcast;
                    bitcast = new BitCastInst(op1, optype, Twine(), 
                                              insertBefore);
                    result = bitcast;
                }
            }
        }
    }
    
    DEBUG(dbgs() << "Convert(\n"
                 << "\top1=" << *op1 << ",\n"
                 << "\topcode=" << Instruction::getOpcodeName(opcode)  << ",\n"
                 << "\toptype=" << *optype  << ")\n"
                 << "returns : " << *result << "\n";);
    
    return result;
}


//===----------------------------------------------------------------------===//
/// IsIdentityTransfrom - check for simple arithmetic iddentity ops.
//===----------------------------------------------------------------------===//

bool OSR::IsIdentityTransfrom(Value *op1, unsigned int opcode, 
                              Type *optype, Value *op2, 
                              Value **Identity) {
    int isIdentity;
    isIdentity = false;
    ConstantInt *constop1;  
    ConstantInt *constop2;  

    if ( optype->isIntegerTy() ) {
        if ( opcode == Instruction::Add ) {
            // check for op1 + 0 or 0 + op2
            if ( isa<ConstantInt>(op1) ) {
                constop1 = dyn_cast<ConstantInt>(op1);
                if ( constop1->isZero() ) {
                    isIdentity = true;
                    if ( Identity != 0 ) {
                        *Identity = op2;
                    }
                    return isIdentity;
                }
            }
            if ( isa<ConstantInt>(op2) ) {
                constop2 = dyn_cast<ConstantInt>(op2);
                if ( constop2->isZero() ) {
                    isIdentity = true;
                    if ( Identity != 0 ) {
                        *Identity = op1;
                    }
                    return isIdentity;
                }
            }
        }
        else if ( opcode == Instruction::Sub ) {
            // check for op1 - 0
            if ( isa<ConstantInt>(op2) ) {
                constop2 = dyn_cast<ConstantInt>(op2);
                if ( constop2->isZero() ) {
                    isIdentity = true;
                    if ( Identity != 0 ) {
                        *Identity = op1;
                    }
                    return isIdentity;
                }
            }
        }
        else if ( opcode == Instruction::Mul ) {
            if ( opcode == Instruction::Add ) {
                // check for op1 * 1 or 1 * op2
                if ( isa<ConstantInt>(op1) ) {
                    constop1 = dyn_cast<ConstantInt>(op1);
                    if ( constop1->isOne() ) {
                        isIdentity = true;
                        if ( Identity != 0 ) {
                            *Identity = op2;
                        }
                        return isIdentity;
                    }
                }
                if ( isa<ConstantInt>(op2) ) {
                    constop2 = dyn_cast<ConstantInt>(op2);
                    if ( constop2->isOne() ) {
                        isIdentity = true;
                        if ( Identity != 0 ) {
                            *Identity = op1;
                        }
                        return isIdentity;
                    }
                }
            }
        }
    }
    return isIdentity;
}


//===----------------------------------------------------------------------===//
/// ExtendConstant - extend the input via one or more 
/// signed/unsigned extensions in Transforms
//===----------------------------------------------------------------------===//

Value *OSR::Extend(Value *op, 
                   std::list<InductionVariable*> Transforms,
                   BasicBlock *header /*=0*/) {
    if ( Transforms.size() > 0 ) {
        if ( isa<ConstantInt>(op) ) {
            return ExtendConstant(dyn_cast<ConstantInt>(op), Transforms);
        }
        else if ( isa<Instruction>(op) )  {
            return ExtendInstruction(op, Transforms, header);
        }
        else if ( isa<Argument>(op) )  {
            return ExtendInstruction(op, Transforms, header);
        }
        else {
            // die another day ...
            return 0;
        }                 
    }
    else {
        return op;
    }
}


//===----------------------------------------------------------------------===//
/// ExtendConstant - extend the input Constant via one or more 
/// signed/unsigned extensions in Transforms
//===----------------------------------------------------------------------===//

Value *OSR::ExtendConstant(ConstantInt *op, 
                           std::list<InductionVariable*> Transforms) {
    int32_t BitWidth;
    ConstantInt *newop;
    ConstantInt *casted;
    std::list<InductionVariable*>::iterator itertrans;
    
    newop = op;
    
    for (itertrans=Transforms.begin(); itertrans!=Transforms.end(); 
         ++itertrans) {
        Transform *trans = *itertrans;
        Type *newoptype = trans->optype;
        DEBUG(dbgs() << "extending op: " << *newop
                     << "\tto type: " << *newoptype 
                     << " via " << Instruction::getOpcodeName(trans->opcode) 
                     << "\n");

        BitWidth = cast<IntegerType>(newoptype)->getBitWidth();
        APInt InitResult(newop->getValue());
        APInt Result;
        if ( trans->opcode == Instruction::ZExt ) {
            Result = InitResult.zext(BitWidth);
        } else if ( trans->opcode == Instruction::SExt ) {
            Result = InitResult.sext(BitWidth);
        }
        casted = ConstantInt::get(newop->getContext(), Result);
        
        newop = casted;
    }
    
    return newop;
}


//===----------------------------------------------------------------------===//
/// ExtendInstruction - extend the input Instruction via one or more 
/// signed/unsigned extensions in Transforms
//===----------------------------------------------------------------------===//

Value *OSR::ExtendInstruction(Value *op, 
                              std::list<InductionVariable*> Transforms,
                              BasicBlock *header /*=0*/) {
    std::list<InductionVariable*>::iterator itertrans;
    Instruction *insertBefore;
    Value *newop;

    insertBefore = 0;
    if ( isa<Argument>(op) && header != 0 ) {
        // definition of op is as an argument
        Argument *arg;
        DomTreeNode *dtnode;
        DomTreeNode *idom;
        arg = dyn_cast<Argument>(op);
        dtnode = DT->getNode(header);
        idom = ( dtnode ? dtnode->getIDom() : 0 );
        if ( idom ) {
            insertBefore = idom->getBlock()->getTerminator();
        }
        else {
            BasicBlock *entryBB;
            entryBB = &(arg->getParent()->getEntryBlock());
            insertBefore = entryBB->getTerminator();
        }
    }
    else {
        insertBefore =  FindApplyInsertLocation(op, 0);
    }

    newop = op;
    
    for (itertrans=Transforms.begin(); itertrans!=Transforms.end(); 
         ++itertrans) {
        Transform *trans = *itertrans;
        Type *newoptype = trans->optype;
        DEBUG(dbgs() << "extending op: " << *newop
                     << "\tto type: " << *newoptype 
                     << " via " << Instruction::getOpcodeName(trans->opcode)
                     << "\n");
        
        if ( trans->opcode == Instruction::ZExt ) {
            ZExtInst *zextinst;
            zextinst = new ZExtInst(op, trans->optype, Twine(),
                                    insertBefore);
            newop = zextinst;
        } else if ( trans->opcode == Instruction::SExt ) {
            SExtInst *sextinst;
            sextinst = new SExtInst(op, trans->optype, Twine(),
                                    insertBefore);
            newop = sextinst;
        }
    }
    
    return newop;
}


//===----------------------------------------------------------------------===//
/// FindApplyInsertLocation - find a location for Apply to insert its new
/// instruction.
//===----------------------------------------------------------------------===//

Instruction *OSR::FindApplyInsertLocation(Value *op1, 
                                          Value *op2) {
    BasicBlock *BB1;
    BasicBlock *BB2;
    BasicBlock *location;
    Instruction *inst1;
    Instruction *inst2;
    Instruction *nearestInst;
    Instruction *insertBefore;
    
    // findNearestCommonDominator
    location = 0;
    
    if ( (op1 && isa<Instruction>(op1)) && 
         (op2 && isa<Instruction>(op2)) ) {
        inst1 = dyn_cast<Instruction>(op1);
        inst2 = dyn_cast<Instruction>(op2);
        BB1 = inst1->getParent();
        BB2 = inst2->getParent();
        // find the op that descends the other and use it's basic block
        if ( BB1 == BB2 ) {
            location = BB1;
            bool doesInst1DominateInst2 = DT->dominates(inst1, inst2);
            nearestInst = ( doesInst1DominateInst2 ? inst2 : inst1 );
        }
        else if ( DT->properlyDominates(BB1, BB2) ) {
            // BB1 properly dominates BB2
            location = BB2;
            nearestInst = inst2;
        }
        else if ( DT->properlyDominates(BB2, BB1) ) {
            // BB2 properly dominates BB1
            location = BB1;
            nearestInst = inst1;
        }
    } else if ( op1 && isa<Instruction>(op1) ) {
        inst1 = dyn_cast<Instruction>(op1);
        location = inst1->getParent();
        nearestInst = inst1;
    } else if ( op2 && isa<Instruction>(op2) ) {
        inst2 = dyn_cast<Instruction>(op2);
        location = inst2->getParent();
        nearestInst = inst2;
    }
    
    if ( location != 0 ) {
        if ( nearestInst->getOpcode() == Instruction::PHI ) {
            insertBefore = nearestInst->getParent()->getFirstNonPHI();
        }
        else {
            insertBefore = nearestInst->getNextNode();
        }
    }
    else {
        // perchance op1 and op2 are globals, arguments or constants
        location = DT->getRoot();
        insertBefore = location->getTerminator();
    }
    
    return insertBefore;
}

//===----------------------------------------------------------------------===//
/// IsInductionVariable - returns true if op is an IV or if op can chain
/// back to IV via a series of zext/sext instructions.
//===----------------------------------------------------------------------===//

bool OSR::IsInductionVariable(Value *op, 
                              Instruction **iv,
                              std::list<InductionVariable*> *Transforms) {
    bool isIV;
    Instruction *instr;
    unsigned int opcode;
    InductionVariable *transform;
    
    *iv = 0;
    
    if ( isa<Instruction>(*op) ) {        
        instr = dyn_cast<Instruction>(op);
        opcode = instr->getOpcode();
        if ( Header[instr] ) {
            isIV = true;
            *iv = instr;
        } else if ( opcode == Instruction::ZExt || 
                    opcode == Instruction::SExt ) {
            // see if op can chain back to an IV 
            // via a series of zext/sext instructions.
            while ( opcode == Instruction::ZExt || 
                    opcode == Instruction::SExt ) {
                op = instr->getOperand(0);
                if ( isa<Instruction>(*op) ) {
                    transform = new InductionVariable(instr->getType());
                    transform->opcode = opcode;
                    transform->ext = instr;
                    Transforms->push_front(transform);                                  
                    instr = dyn_cast<Instruction>(op);
                    opcode = instr->getOpcode();
                } else {
                    instr = 0;
                    break;
                }
            }
            if ( instr != 0 && Header[instr] ) {
                isIV = true;
                *iv = instr;
            } else {
                isIV = false;
            }

        } else {
            isIV = false;
        }
    } else {
        isIV = false;
    }
    
    if ( !isIV && Transforms->size() > 0 ) {
        FreeIVs(Transforms);
    }
    
    return isIV;
}


//===----------------------------------------------------------------------===//
/// CrackInstruction - slices and dices an instruction into its induction
/// variable and regions constant parts.
//===----------------------------------------------------------------------===//

OSR::Cracked OSR::CrackInstruction(Instruction *I) {
/*
 n is of the form x ← iv × rc, x ← rc × iv, 
                  x ← iv ± rc, or x ← rc + iv ?
 */
    Cracked cracked;
    bool hasProperForm;
    unsigned int opcode;
    Value *op1;
    Value *op2;
    Value *op3;
    Instruction *op1iv;
    Instruction *op2iv;
    Instruction *op3iv;
    int NumOperands;
    bool isOp1RegionConstant;
    bool isOp2RegionConstant;
    bool isOp1InductionVariable;
    bool isOp2InductionVariable;
    bool isOp3InductionVariable;
    std::list<InductionVariable*> op1Transforms; 
    std::list<InductionVariable*> op2Transforms; 
    std::list<InductionVariable*> op3Transforms; 
    
    hasProperForm = false;
    cracked.IV = 0;
    cracked.RC = 0;
    op1Transforms.clear();
    op2Transforms.clear();
    op3Transforms.clear();
    
    opcode = I->getOpcode();
    
    if ( I->isBinaryOp() ) {
        
        op1 = I->getOperand(0);
        op2 = I->getOperand(1);
        
        //isOp1InductionVariable = ( isa<Instruction>(*op1) && 
        //                           Header[dyn_cast<Instruction>(op1)] );
        isOp1InductionVariable = IsInductionVariable(op1, 
                                                     &op1iv, 
                                                     &op1Transforms);
        //isOp2InductionVariable = ( isa<Instruction>(*op2) && 
        //                           Header[dyn_cast<Instruction>(op2)] );
        isOp2InductionVariable = IsInductionVariable(op2, 
                                                     &op2iv, 
                                                     &op2Transforms);
        
        // Header[] must refer to the IV's header BB
        
        isOp1RegionConstant = 
            ( isOp2InductionVariable
                ? RegionConst(op1, Header[dyn_cast<Instruction>(op2)])
              //: isa<Constant>(op1) );
                : false );
        isOp2RegionConstant = 
            ( isOp1InductionVariable
                ? RegionConst(op2, Header[dyn_cast<Instruction>(op1)]) 
              //: isa<Constant>(op2) );
                : false );
        
        if ( opcode == Instruction::Add ) {
            // x ← iv + rc, or x ← rc + iv
            if      ( isOp1RegionConstant && isOp2InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op2iv; // dyn_cast<Instruction>(op2);
                cracked.transforms = op2Transforms;
                cracked.RC = op1;
                cracked.isOp1IV = false;
            } 
            else if ( isOp2RegionConstant && isOp1InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op1iv; // dyn_cast<Instruction>(op1);
                cracked.transforms = op1Transforms;
                cracked.RC = op2;
                cracked.isOp1IV = true;
            }
        } 
        else if ( opcode == Instruction::Mul ) {
            // x ← iv × rc, x ← rc × iv
            if      ( isOp1RegionConstant && isOp2InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op2iv; // dyn_cast<Instruction>(op2);
                cracked.transforms = op2Transforms;
                cracked.RC = op1;
                cracked.isOp1IV = false;
            } 
            else if ( isOp2RegionConstant && isOp1InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op1iv; // dyn_cast<Instruction>(op1);
                cracked.transforms = op1Transforms;
                cracked.RC = op2;
                cracked.isOp1IV = true;
            }
        } 
        else if ( opcode == Instruction::Sub ) {
            // x ← iv - rc
            if ( isOp2RegionConstant && isOp1InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op1iv; // dyn_cast<Instruction>(op1);
                cracked.transforms = op1Transforms;
                cracked.RC = op2;
                cracked.isOp1IV = true;
            }
        }
    }
    else if ( opcode == Instruction::GetElementPtr ) {
        // look for getelementptr RC, IV
        GetElementPtrInst *GEP;
        
        GEP = dyn_cast<GetElementPtrInst>(I);
        NumOperands = GEP->getNumOperands();
        if ( NumOperands == 2 ) {
            op1 = GEP->getOperand(0); // llvm::PointerType
            op2 = GEP->getOperand(1);
            
            //isOp2InductionVariable = ( isa<Instruction>(*op2) && 
            //                           Header[dyn_cast<Instruction>(op2)] );
            isOp2InductionVariable = IsInductionVariable(op2, 
                                                         &op2iv,
                                                         &op2Transforms);
            isOp1InductionVariable = IsInductionVariable(op1, 
                                                         &op1iv,
                                                         &op1Transforms);
            if ( isOp2InductionVariable ) {
                isOp1RegionConstant = 
                  RegionConst(op1, Header[dyn_cast<Instruction>(op2)]);
            }
            else {
                isOp1RegionConstant = false;
            }
            if ( isOp1InductionVariable ) {
                isOp2RegionConstant = 
                  RegionConst(op2, Header[dyn_cast<Instruction>(op1)]);
            }
            else {
                isOp2RegionConstant = false;
            }

            if ( isOp1RegionConstant && isOp2InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op2iv; // dyn_cast<Instruction>(op2);
                cracked.transforms = op2Transforms;
                cracked.RC = op1;
                cracked.isOp1IV = false;
            }
            else if ( isOp2RegionConstant && isOp1InductionVariable ) {
                cracked.foundCandidate = true;
                cracked.opcode = opcode;
                cracked.IV = op1iv; // dyn_cast<Instruction>(op2);
                cracked.transforms = op1Transforms;
                cracked.RC = op2;
                cracked.isOp1IV = true;
            }
        } else if ( NumOperands == 3 ) {
            gep_type_iterator GTI = gep_type_begin(GEP);
            Type *gettype1 = *GTI;
            GTI++;
            Type *gettype2 = *GTI;
            if ( gettype1->isPointerTy() && 
                 (gettype2->isArrayTy() || gettype2->isPointerTy()) ) {
                op1 = GEP->getOperand(0);
                op2 = GEP->getOperand(1);
                op3 = GEP->getOperand(2);
                ConstantInt *constop2 = dyn_cast<ConstantInt>(op2);
                if ( constop2 != 0 && constop2->isZero() ) {
                    //isOp3InductionVariable = ( isa<Instruction>(*op3) && 
                    //                           Header[dyn_cast<Instruction>(op3)] );
                    isOp3InductionVariable = IsInductionVariable(op3, 
                                                                 &op3iv,
                                                               &op3Transforms);
                    if ( isOp3InductionVariable ) {
                        isOp1RegionConstant = 
                          RegionConst(op1, Header[dyn_cast<Instruction>(op3)]);
                    }
                    else {
                        isOp1RegionConstant = false;
                    }
                    if ( isOp1RegionConstant && isOp3InductionVariable ) {
                        cracked.foundCandidate = true;
                        cracked.opcode = opcode;
                        cracked.IV = op3iv; // dyn_cast<Instruction>(op3);
                        cracked.transforms = op3Transforms;
                        cracked.RC = op1;
                        cracked.isOp1IV = false;
                    }
                }
            }
        }        
    }
#if 0
    else if ( opcode == Instruction::SExt ||
             opcode == Instruction::ZExt) {
        // x ← sign extension iv
        op1 = I->getOperand(0);
        
        //isOp1InductionVariable = ( isa<Instruction>(*op1) && 
        //                           Header[dyn_cast<Instruction>(op1)] );
        isOp1InductionVariable = IsInductionVariable(op1, &op1iv);
        
        if ( isOp1InductionVariable ) {
            cracked.foundCandidate = true;
            cracked.opcode = opcode;
            cracked.IV = dyn_cast<Instruction>(op1);
            cracked.RC = 0;
            cracked.isOp1IV = true;
        }
    }
    else if ( opcode == Instruction::BitCast ) {
        // x ← bit cast iv
        op1 = I->getOperand(0);
        
        //isOp1InductionVariable = ( isa<Instruction>(*op1) && 
        //                          Header[dyn_cast<Instruction>(op1)] );
        isOp1InductionVariable = IsInductionVariable(op1, 
                                                     &op1iv, 
                                                     &op1Transforms);
        
        if ( isOp1InductionVariable ) {
            cracked.foundCandidate = true;
            cracked.opcode = opcode;
            cracked.IV = op1iv; // dyn_cast<Instruction>(op1);
            cracked.transforms = op1Transforms;
            cracked.RC = 0;
            cracked.isOp1IV = true;
        }
    }
#endif
    
    if ( !cracked.foundCandidate ) {
        FreeIVs(&op1Transforms);
        FreeIVs(&op2Transforms);
        FreeIVs(&op3Transforms);
    }
    
    return cracked;
}


//===----------------------------------------------------------------------===//
/// RPO - give each Basic Block a Reverse Postorder (RPO) number.
//===----------------------------------------------------------------------===//

void OSR::RPO(BasicBlock *BB) {
    BBVisited[BB] = true;
    for (succ_iterator SI = succ_begin(BB), SE = succ_end(BB); SI != SE; ++SI) 
    {
        if ( !BBVisited[*SI] ) {
            RPO(*SI);
        }
    }
    RPOnum[BB] = nextDFSnum--;
    DEBUG(dbgs() << "Assigning RPO #" << RPOnum[BB] << " to basic block: " 
                 << BB->getName() << "\n";);
}

//===----------------------------------------------------------------------===//
/// ReplaceInstruction - Need to back track and replace an instruction 
/// previously visited via OSR. The old instruction is deleted. The new
/// instruction has been added to the same basic back as the old.
//===----------------------------------------------------------------------===//

void OSR::ReplaceInstruction(Instruction *OldInst, 
                             Instruction *NewInst) {
    DEBUG(
        char hexI[1024];
        snprintf(hexI, 1024, "%#.16lx", (unsigned long)OldInst);
        dbgs() << "ReplaceInstruction: deleting instructon:" << hexI << "\n"
               << *OldInst << "\n"
               << "from basic block: " << OldInst->getParent()->getName() << "\n";
        int size = Nodes.size();
        int i;
        for (i=0; i<size; i++) {
            if ( Nodes[i] == OldInst ) {
                dbgs() << "HOUSTON WE HAVE A PROBLEM!!!\n";
            }
        }
    );
    
    if ( Header.count(OldInst) ) {
        Header[NewInst]  = Header[OldInst];
        Header.erase(OldInst);
    }
    if ( Visited.count(OldInst) ) {
        Visited[NewInst] = Visited[OldInst];
        Visited.erase(OldInst);
    }
    if ( DFSnum.count(OldInst) ) {
        DFSnum[NewInst]  = DFSnum[OldInst];
        DFSnum.erase(OldInst);
    }
    SCCs.ReplaceInst(OldInst, NewInst);

    UpdateHashTable(OldInst, NewInst);
    
    BasicBlock::iterator ii(OldInst);
    ReplaceInstWithValue(OldInst->getParent()->getInstList(), ii, NewInst);    
}

//===----------------------------------------------------------------------===//
/// ReplaceInstruction - Need to replace all uses of the old instruction
/// to uses of the new instruction.  The old instruction is deleted. The new
/// instruction may be defined in a different basic block.
//===----------------------------------------------------------------------===//

void OSR::ReplaceInstructionUse(Instruction *OldInst, 
                                Instruction *NewInst) {
    DEBUG(
        char hexI[1024];
        snprintf(hexI, 1024, "%#.16lx", (unsigned long)OldInst);
        dbgs() << "ReplaceInstructionUse: deleting instructon:" << hexI << "\n"
               << *OldInst << "\n"
             << "from basic block: " << OldInst->getParent()->getName() << "\n";
        int size = Nodes.size();
        int i;
        for (i=0; i<size; i++) {
            if ( Nodes[i] == OldInst ) {
                dbgs() << "HOUSTON WE HAVE A PROBLEM!!!\n";
            }
        }
    );
    
    if ( Header.count(OldInst) ) {
        Header.erase(OldInst);
    }
    if ( Visited.count(OldInst) ) {
        Visited.erase(OldInst);
    }
    if ( DFSnum.count(OldInst) ) {
        DFSnum.erase(OldInst);
    }
    SCCs.RemoveInst(OldInst);
    
    UpdateHashTable(OldInst, NewInst);
    
    BasicBlock::iterator ii(OldInst);
    ReplaceInstWithValue(OldInst->getParent()->getInstList(), ii, NewInst);    
}

//===----------------------------------------------------------------------===//
/// ReplaceInductionVariable - Replace an Induction Variable
//===----------------------------------------------------------------------===//

void OSR::ReplaceInductionVariable(Instruction *OldInst, 
                                   Instruction *NewInst) {
    DEBUG(
          char hexI[1024];
          snprintf(hexI, 1024, "%#.16lx", (unsigned long)OldInst);
          dbgs() << "ReplaceInductionVariable: deleting instructon:" << hexI << "\n"
          << *OldInst << "\n"
          << "from basic block: " << OldInst->getParent()->getName() << "\n";
          int size = Nodes.size();
          int i;
          for (i=0; i<size; i++) {
              if ( Nodes[i] == OldInst ) {
                  dbgs() << "HOUSTON WE HAVE A PROBLEM!!!\n";
              }
          }
          );
    
    if ( Header.count(OldInst) ) {
        Header[NewInst] = Header[OldInst];
        Header.erase(OldInst);
    }
    if ( Visited.count(OldInst) ) {
        Visited[NewInst] = Visited[OldInst];
        Visited.erase(OldInst);
    }
    if ( DFSnum.count(OldInst) ) {
        DFSnum[NewInst] = DFSnum[OldInst];
        DFSnum.erase(OldInst);
    }
    SCCs.ReplaceInst(OldInst, NewInst);
    
    UpdateHashTable(OldInst, NewInst);
    
    BasicBlock::iterator ii(OldInst);
    ReplaceInstWithValue(OldInst->getParent()->getInstList(), ii, NewInst);    
}

//===----------------------------------------------------------------------===//
/// DeleteInstruction - Need to delete an instruction
//===----------------------------------------------------------------------===//

void OSR::DeleteInstruction(Instruction *OldInst) {
    DEBUG(
        char hexI[1024];
        snprintf(hexI, 1024, "%#.16lx", (unsigned long)OldInst);
        dbgs() << "DeleteInstruction: deleting instructon:" << hexI << "\n"
               << *OldInst << "\n"
             << "from basic block: " << OldInst->getParent()->getName() << "\n";
        int size = Nodes.size();
        int i;
        for (i=0; i<size; i++) {
            if ( Nodes[i] == OldInst ) {
                dbgs() << "HOUSTON WE HAVE A PROBLEM!!!\n";
            }
        }
    );
    
    if ( Header.count(OldInst) ) {
        Header.erase(OldInst);
    }
    if ( Visited.count(OldInst) ) {
        Visited.erase(OldInst);
    }
    if ( DFSnum.count(OldInst) ) {
        DFSnum.erase(OldInst);
    }
    SCCs.RemoveInst(OldInst);
    
    UpdateHashTable(OldInst, 0);
    
    OldInst->eraseFromParent();
}

//===----------------------------------------------------------------------===//
/// DumpHashTable - dump hash table, debug only.
//===----------------------------------------------------------------------===//

void OSR::DumpHashTable(const char *label) {
    unsigned int j;
    unsigned int size;
    char value_str[128];
    char key[129];

    size = AddedInstructions.size();
    j = 0;
    for (std::map<MultiKey, Instruction*>::iterator i 
         = AddedInstructions.begin(); j < size; ++i, ++j) {
        MultiKey multikey = i->first;
        Instruction *value = i->second;

        const Value *IV           = multikey.key1;
        const unsigned int opcode = multikey.key2;
        Type *optype              = multikey.key3;
        const Value *RC           = multikey.key4;

        BuildHashKey(IV, opcode, optype, RC, key, 128);
        snprintf(value_str, 128, "%#.16lx",  (unsigned long)value);

        dbgs() << "entry#" << j << " = key('" << key << "'), value(" 
               << value_str << ")\n";
#if 0
        dbgs() << "entry#" << j << " = \n" 
               << "\tIV=" << *IV << "\n"
               << "\topcode=" << Instruction::getOpcodeName(opcode)  << "\n"
               << "\toptype=" << *optype  << "\n";
        if ( RC != 0 ) {
            dbgs() << "\tRC=" << *RC << "\n";
        } else {
            dbgs() << "\tRC=nada\n";
        }
        dbgs() << "\tmaps to value=" << *value << "\n";
#endif
    }
}


//===----------------------------------------------------------------------===//
/// SearchHashTable - search hash table to see if we have already reduced
/// this instruction.
//===----------------------------------------------------------------------===//

Instruction* OSR::SearchHashTable(const Value *IV, 
                                  const unsigned int opcode,
                                  Type *optype,
                                  const Value *RC) {
    Instruction *result;
        
    MultiKey multikey = MultiKey(IV, opcode, optype, RC);
    result = AddedInstructions[multikey];
    
    DEBUG(
        char *key = new char[128];
        BuildHashKey(IV, opcode, optype, RC, key, 128);
        dbgs() << "searching hash table with key('" << key << "') had a "
               << ( result == 0 ? "miss\n" : "hit\n" );
        if ( result == 0 ) {
            //DEBUG(DumpHashTable("after a miss on hash table search"););
        }
        delete key; key = 0;
    );
    
    return result;
}


//===----------------------------------------------------------------------===//
/// AddHashTable - add an entry into the hash table.
//===----------------------------------------------------------------------===//

void OSR::AddHashTable(const Value *IV, 
                       const unsigned int opcode, 
                       Type *optype,
                       const Value *RC, 
                       Instruction *inst) {
    const MultiKey multikey = MultiKey(IV, opcode, optype, RC);
    AddedInstructions[multikey] = inst;
    
    DEBUG(DumpHashTable("after adding"););
}


//===----------------------------------------------------------------------===//
/// UpdateHashTable - swap values within hash table.
//===----------------------------------------------------------------------===//

void OSR::UpdateHashTable(Value *oldValue, 
                          Value *newValue) {
    // snprintf(key, keysize, "%#.16lx %#.8x %#.16lx", 
    //          (const unsigned long)IV, opcode, (const unsigned long)RC);
    int j;
    int size;
    const int keysize = 128;
    char key[keysize];
    char newkey[keysize];
    //std::string keystr;
    //Instruction *value;
    
    DEBUG(snprintf(newkey, keysize, "%#.16lx", 
                   (const unsigned long)oldValue);
          dbgs() << "Looking into the Hash Table for key:\n"
                 << "\t" << newkey << "\n";);

    size = AddedInstructions.size();
    j = 0;
    for (std::map<MultiKey, Instruction*>::iterator i = AddedInstructions.begin();
         j < size; ++i, ++j) {
        MultiKey multikey = i->first;
        Instruction *value = i->second;
        
        // ignore the empty hash table entries
        if ( value == 0 ) continue;
        
        const Value *IV           = multikey.key1;
        const unsigned int opcode = multikey.key2;
        Type *optype              = multikey.key3;
        const Value *RC           = multikey.key4;
        
        if ( isa<Instruction>(oldValue) && 
             newValue != 0 && isa<Instruction>(newValue) ) {
            if ( (void*)value == (void*)oldValue ) {
                DEBUG(BuildHashKey(IV, opcode, optype, RC, key, 128);
                      dbgs() << "Updating the hash entry for key " << key <<"\n"
                             << "\treplace value: " << *oldValue << "\n"
                             << "\twith value: " << *newValue << "\n";);
                AddedInstructions[multikey] = dyn_cast<Instruction>(newValue);
            }
        }
        else if ( isa<Instruction>(oldValue) && newValue == 0 ) {
            if ( (void*)value == (void*)oldValue ) {
                DEBUG(BuildHashKey(IV, opcode, optype, RC, key, 128);
                      dbgs() << "Deleting the hash entry for key " << key <<"\n"
                             << "\tand value: " << *oldValue << "\n";);
                AddedInstructions.erase(multikey);
            }
        }
        
        if ( oldValue == IV ) {
            if ( newValue != 0 ) {
                MultiKey newmultikey = MultiKey(newValue, opcode, optype,RC);
                DEBUG(BuildHashKey(oldValue, opcode, optype, RC, key, 128);
                      dbgs() << "Replacing key " << key;
                      BuildHashKey(newValue, opcode, optype, RC, key, 128);
                      dbgs() << " with new key " << newkey << "\n";);
                AddedInstructions[newmultikey] = AddedInstructions[multikey];
                AddedInstructions.erase(multikey);
            }
            else {
                DEBUG(BuildHashKey(IV, opcode, optype, RC, key, 128);
                      dbgs() << "Deleting the hash entry for key " << key <<"\n"
                             << "\tand value: " << *oldValue << "\n";);
                AddedInstructions.erase(multikey);
            }
        }
        
        if ( oldValue == RC ) {
            if ( newValue != 0 ) {
                MultiKey newmultikey = MultiKey(IV, opcode, optype, newValue);
                DEBUG(BuildHashKey(IV, opcode, optype, oldValue, key, 128);
                      dbgs() << "Replacing key " << key;
                      BuildHashKey(IV, opcode, optype, newValue, key, 128);
                      dbgs() << " with new key " << newkey << "\n";);
                AddedInstructions[newmultikey] = AddedInstructions[multikey];
                AddedInstructions.erase(multikey);
            }
            else {
                DEBUG(BuildHashKey(IV, opcode, optype, RC, key, 128);
                      dbgs() << "Deleting the hash entry for key " << key <<"\n"
                             << "\tand value: " << *oldValue << "\n";);
                AddedInstructions.erase(multikey);
            }
        }
    }
    
    DEBUG(DumpHashTable("UpdateHashTable()"););
}


void OSR::SetNoWrap(Instruction * i, Instruction *iv) {
    int opcode;
    InductionVariable *IV;
    BinaryOperator *binop;

    if ( i->isBinaryOp() ) {
        opcode = i->getOpcode();
        if ( opcode == Instruction::Add || opcode == Instruction::Sub ) {
            // binop->hasNoSignedWrap() || binop->hasNoUnsignedWrap()
            IV = FindIV(InductionVariables, SCCs.FindSCC(iv));
            if ( IV ) {
                binop = dyn_cast<BinaryOperator>(i);
                if ( IV->noSignedWrap ) {
                    binop->setHasNoSignedWrap(true);
                }
                if ( IV->noUnsignedWrap ) {
                    binop->setHasNoUnsignedWrap(true);
                }
            }
        }
    }
}


//===----------------------------------------------------------------------===//
/// LFTR - Linear Function Test Replacement pass.
//===----------------------------------------------------------------------===//


void OSR::ScoreReplacement(InductionVariable *IV) {
    unsigned int lowchildscore;
    std::list<InductionVariable*> children;
    std::list<InductionVariable*>::iterator ichild;
    InductionVariable *child;
    
    lowchildscore = 0xFFFFFFFF;
    if ( IV->children.size() > 0 ) {
        children = IV->children;
        for (ichild=children.begin(); ichild!=children.end(); ++ichild) {
            child = *ichild;
            ScoreReplacement(child);
            if ( child->score < lowchildscore ) {
                lowchildscore = child->score;
            }
        }
    }
    
    IV->score = lowchildscore + 1;
}


void OSR::FindReplacements(InductionVariable *IV,
                           std::list<InductionVariable*> *Replacements) {
    unsigned int lowchildscore;
    std::list<InductionVariable*> children;
    std::list<InductionVariable*>::iterator ichild;
    InductionVariable *child;
    InductionVariable *lowchild;

    lowchildscore = 0xFFFFFFFF;
    if ( IV->children.size() > 0 ) {
        // find IV's lowest cost child
        children = IV->children;
        for (ichild=children.begin(); ichild!=children.end(); ++ichild) {
            child = *ichild;
            if ( child->score < lowchildscore ) {
                lowchildscore = child->score;
                lowchild = child;
            }
        }
        // add the lowest cost child to the list and recurse.
        Replacements->push_back(lowchild);
        FindReplacements(lowchild, Replacements);
    }
}


void OSR::LFTR(Function& F) {
    std::list<InductionVariable*>::iterator i;
    InductionVariable *IV;
    InductionVariable *nextIV;
    InductionVariable *lastIV;
    Value::use_iterator ui;
    Value::use_iterator ubegin;
    Value::use_iterator uend;
    Value *use;
    Instruction *outside_use;
    Instruction *inst;
    Instruction *firstuse;
    PHINode *masterphi;
    PHINode *phi;
    ICmpInst *icmp;
    ICmpInst *newcmp;
    bool failsConstraints;
    bool foundCandidate;
    bool isOp1IV;
    bool replaceTest;
    Value *op1;
    Value *op2;
    Value *noniv;
    Value *nextnoniv;
    Value *currentnoniv;
    Value *iv;
    Value *currentiv;
    std::list<InductionVariable*> NoTransforms;
    CmpInst::Predicate pred;
    std::list<InductionVariable*> Replacements;
    std::list<InductionVariable*>::iterator irep;
    std::vector<Instruction*> scc;
    std::vector<Instruction*>::iterator scci;
    std::vector<Instruction*>::iterator scc_begin;
    std::vector<Instruction*>::iterator scc_end;
    Instruction *scc_inst;
    unsigned int uses_outside_scc;
    bool DeletePHIIfEmpty;
    int NumOperands;
    int j;
    bool changed;
    int SCCId;
    
    // revese sort InductionVariables via the RPO# of IV's basic block
    // InductionVariables will then be in post order => inner loops
    // handled before outer loops.
    
    DEBUG(dbgs() << "...sorting the induction variables by RPO number...\n";);
    InductionVariables.sort(CompareIVs);
    DEBUG(DumpIVs(InductionVariables););

    do {
        replaceTest = false;

        for (i=InductionVariables.begin(); i!=InductionVariables.end(); ++i) {
            IV = *i;

            // phi is the unique key of the induction variable
            if ( IV->doneWithLFTR ) continue; // test already replaced
            
            SCCId = IV->SCCId;
            masterphi = SCCs.GetHeadPhi(SCCId);
            
            // if we have not created induction variables from this one,
            // go to the next one
            if ( IV->children.size() == 0 ) continue;
            
            DEBUG(dbgs() << "\n"
                         << "examining IV candidate for test replacement:" 
                         << *masterphi << "\n";);
            scc = SCCs.GetSCC(SCCs.FindSCC(masterphi)); //FindSCC(masterphi);
            scc_begin = scc.begin();
            scc_end   = scc.end();
            failsConstraints = false;
            for (scci = scc_begin; scci != scc_end; ++scci) {
                scc_inst = *scci;
                DEBUG(dbgs() << "\tSCC node: " << *scc_inst << "\n";);
                
                uses_outside_scc = 0;
                outside_use = 0;
                ubegin = scc_inst->use_begin();
                uend   = scc_inst->use_end();
                for (ui=ubegin; ui !=uend; ++ui) {
                    use = *ui;
                    DEBUG(dbgs() << "\t\tUse: " << *use << "\n";);

                    if ( isa<Instruction>(use) ) {
                        inst = dyn_cast<Instruction>(use);
                        if ( SCCs.GetHeadPhi(SCCs.FindSCC(inst)) == masterphi ) {
                            // this use of the SCC node is within
                            // the SCC.
                        } else {
                            outside_use = inst;
                            uses_outside_scc++;
                        }
                    }
                }
                
                if ( scc_inst == masterphi ) {
                    // to qaulify for a LFFTR candidte, the master phi
                    // must have one and only one use outside of the SCC:
                    // a compare instruction
                    if ( uses_outside_scc == 1 ) {
                        if ( outside_use->getOpcode() == Instruction::ICmp ) {
                            // Zeus smiles from on high
                            icmp = dyn_cast<ICmpInst>(outside_use);
                        }
                        else {
                            DEBUG(dbgs() << "\tone and only outside use of the "
                                  << "master phi must be an icmp: FAIL.\n";);
                            failsConstraints = true;
                        }
                    }
                    else {
                        DEBUG(dbgs() << "\tmust have one and only use outside "
                                     << "of the SCC: FAIL.\n";);
                        failsConstraints = true;
                    }
                }
                else {
                    // all other SCC node uses must be contained within
                    // the SCC (or we would have dangling uses after
                    // deleting all of the SCC nodes)
                    if ( uses_outside_scc == 0 ) {
                        // Zeus smiles from on high
                    } else {
                        DEBUG(dbgs() << "\thas more than one use: FAIL.\n";);
                        failsConstraints = true;
                    }
                }
            }
#if 0
            Instruction *firstandonlyuse;
            int use_count;
            int cmp_count;
            int phi_count;
            ui = phi->use_begin();
            uend = phi->use_end();
            use_count = 0;
            cmp_count = 0;
            phi_count = 0;
            for (; ui !=uend; ++ui) {
                use = *ui;
                DEBUG(dbgs() << "use#" << use_count << " : " << *use << "\n";);
                if ( isa<Instruction>(use) ) {
                    inst = dyn_cast<Instruction>(use);
                    if ( SCCPHI[inst] == phi ) {
                        DEBUG(dbgs() << "\tis part of the SCC.\n";);
                        // must have only a single use in a defintion
                        // which is another instruction in the SCC
                        if ( inst->hasOneUse() ) {
                            // must have only one and only one use
                            DEBUG(dbgs() << "\thas only one use.\n";);
                            Instruction *firstandonlyuse = 
                              dyn_cast<Instruction>(*inst->use_begin());
                            if  ( SCCPHI[firstandonlyuse] == phi ) {
                                // the one and only use is in the SCC
                                if ( inst->getOpcode() == Instruction::PHI ) {
                                    // can only have one phi per SCC
                                    // any phi use of the master phi is a no go
                                    DEBUG(dbgs() << "\tmore than one PHI"
                                          << ": FAIL\n";);
                                    failsConstraints = true;
#if 0
                                    phi_count++;
                                    if ( phi_count > 1 ) {
                                        DEBUG(dbgs() << "\tmore than one PHI"
                                                     << ": FAIL\n";);
                                        failsConstraints = true;
                                    }
#endif
                                }
                            }
                            else {
                                DEBUG(dbgs() << "\tfirst and only is use is not"
                                             << " in the SCC: FAIL.\n";);
                                failsConstraints = true;
                            }
                        }
                        else {
                            DEBUG(dbgs() << "\thas more than one use: "
                                         << "FAIL.\n";);
                            failsConstraints = true;
                        }

                    } else if ( inst->getOpcode() == Instruction::ICmp ) {
                        DEBUG(dbgs() << "\tfound icmp\n";);
                        icmp = dyn_cast<ICmpInst>(inst);
                        cmp_count++;
                        if ( cmp_count > 1 ) {
                            DEBUG(dbgs() << "\tfound more than one icmp: "
                                         << "FAIL.\n";);
                            failsConstraints = true;
                        }
                    } else {
                        DEBUG(dbgs() << "\tis not in SCC and is not icmp: "
                                     << "FAIL.\n";);
                        failsConstraints = true;
                    }
                }
                else {
                    DEBUG(dbgs() << "\tis not instruction: FAIL.\n";);
                    failsConstraints = true;
                }
                use_count++;
            }
#endif

            // to be a candidate, iv uses must stay within the SCC, except
            // for a single icmp instruction.  removing the icmp will make
            // the iv's SCC useless and deletable.
            foundCandidate = ( failsConstraints == false );
            if ( foundCandidate ) {
                // take the icmp candidate:
                //   icmp iv, noniv (or icmp noniv, iv)
                // perform one or more xforms on the icmp candidate operands.
                // create a new icmp instruction and replace the old one.
                // note that xforming the iv opreand has already been done
                // and all that needs to be done is to tranform the noniv 
                // operand. the xformed iv operand is stored in the global
                // InductionVariables tree.

                DEBUG(dbgs() << "found a candidate for linear funcion "
                             << "test replacement.\n";);
                
                op1 = icmp->getOperand(0);
                op2 = icmp->getOperand(1);
                SCCId = IV->SCCId;
                isOp1IV = ( isa<Instruction>(op1) && 
                            dyn_cast<Instruction>(op1) == SCCs.GetHeadPhi(SCCId) );
                noniv = ( isOp1IV ? op2 : op1 );
                iv    = ( isOp1IV ? op1 : op2 );
                firstuse = dyn_cast<Instruction>(*(icmp->use_begin()));
                
                ScoreReplacement(IV);
                Replacements.clear();
                FindReplacements(IV, &Replacements);

                currentnoniv = noniv;
                for (irep = Replacements.begin(); irep != Replacements.end(); 
                     ++irep) {
                    nextIV = *irep;
                    DEBUG(if ( nextIV->doneWithLFTR )
                            dbgs() << "next iv : nada\n";
                          else
                            dbgs() << "next iv : " << *SCCs.GetHeadPhi(nextIV->SCCId) << "\n";);
                    // calling Apply() can create new IVs.  do not want,
                    // so call ApplyNoReduce() instead.
                    nextnoniv = ApplyNoReduce(currentnoniv,
                                              nextIV->opcode, nextIV->optype,
                                              nextIV->RC, NoTransforms);
                    DEBUG(dbgs() << "ApplyNoReduce(\n"
                                 << "\top1=" << *currentnoniv << ",\n"
                                 << "\topcode=" << Instruction::getOpcodeName(nextIV->opcode)  << ",\n"
                                 << "\toptype=" << *nextIV->optype  << ",\n"
                                 << "\top2=" << *nextIV->RC << ")\n"
                                 << "returns : " << *nextnoniv << "\n";);
                    currentnoniv = nextnoniv;
                }
                lastIV = nextIV;
                
                currentiv = SCCs.GetHeadPhi(lastIV->SCCId);
                
                // icmp is the instruction that we need to replace
                // create a new compare:
                //      %12 = icmp ugt float* %6, %7
                if ( currentiv->getType()->isPointerTy() ) {
                    pred = icmp->getUnsignedPredicate();
                }
                else {
                    pred = ( icmp->isSigned()
                               ? icmp->getSignedPredicate()
                               : icmp->getUnsignedPredicate() );
                }

                if ( isOp1IV ) {
                    newcmp = new ICmpInst(firstuse, pred, 
                                          currentiv, currentnoniv);
                } else {
                    newcmp = new ICmpInst(firstuse, pred, 
                                          currentnoniv, currentiv);
                }

                DEBUG(dbgs() << "deleting old compare:\n"
                             << *icmp << "\n";);
                
                // replace old icmp with new icmp
                ReplaceInstruction(icmp, newcmp);
                
                DEBUG(dbgs() << "adding new compare:\n"
                             << *newcmp << "\n";);
                
                // 
                // Clean up now instead of later with -instcombine
                //
                
                // remove all of operand from the phi instructions 
                // withi the SCC.

                scc_begin = scc.begin();
                scc_end   = scc.end();
                for (scci = scc_begin; scci != scc_end; ++scci) {
                    scc_inst = *scci;
                    if ( isa<PHINode>(scc_inst) ) {
                        phi = dyn_cast<PHINode>(scc_inst);
                        NumOperands = phi->getNumOperands() / 2;
                        DeletePHIIfEmpty = false;
                        for (j=0; j<NumOperands; j++) {
                            phi->removeIncomingValue(unsigned(0), 
                                                     DeletePHIIfEmpty);
                        }
                    }
                }
                
                // remove all of the nodes that have zero used from the SCC.
                // rinse. repeat.

                do {
                    changed = false;
                    scc_begin = scc.begin();
                    scc_end   = scc.end();
                    for (scci = scc_begin; scci != scc_end; ++scci) {
                        scc_inst = *scci;
                        if ( scc_inst == 0 ) continue;
                        
                        if ( scc_inst->getNumUses() == 0 ) {
                            DEBUG(dbgs() << "removing: " << *scc_inst << "\n");
                            DeleteInstruction(scc_inst);
                            changed = true;
                            *scci = 0;
                        }
                    }
                } while (changed);
#if 0
                // delete iv->phi (aka phi) and all its uses
                bool DeletePHIIfEmpty;
                int NumOperands, j;
                DeletePHIIfEmpty = false;
                NumOperands = masterphi->getNumOperands() / 2;
                for (j=0; j<NumOperands; j++) {
                    masterphi->removeIncomingValue(unsigned(0), 
                                                   DeletePHIIfEmpty);
                }
                
                ui = masterphi->use_begin();
                uend = masterphi->use_end();
                for (; ui !=uend; ++ui) {
                    use = *ui;
                    inst = dyn_cast<Instruction>(use);
                    DEBUG(dbgs() << "deleting : " << *inst << "\n";);
                    inst->eraseFromParent();
                }

                DEBUG(dbgs() << "deleting : " << *masterphi << "\n";);
                masterphi->eraseFromParent();
#endif
                NumInductionVariableRemoved++;
                NumInductionVariableFound--;

                DEBUG(DumpIR(F););

                IV->doneWithLFTR = true;
                replaceTest = true;
            }
        }
    } while ( replaceTest );
    
    FreeIVs(&InductionVariables);
}


