/* 
 ##  class FPoBand
 ##
 ##  This class contains the routines for positive definite matrices
 ##  in banded form.
 ##
 */

package edu.rice.linpack.Matrix.FMatrix;
import edu.rice.linpack.LNumber.*;
import edu.rice.linpack.util.*;
import edu.rice.linpack.Vector.*;

public class FPoBand extends FBanded {
  
  private int M;

  public FPoBand() {
    super();
  }
  public  FPoBand(int c, int M2) {
    super(0,M2,c);
    mu = M2;
    ml = 0;
    M = M2;
  }
  public FPoBand(float[][] f, int ML, int MU) {
    mu = MU;
    M = mu;
    ml = 0;
    rows = M+1;
    cols = f[0].length;
    Mat = band(f);
  }
  public FPoBand(float[][] f, int M2) {
    mu = M2;
    M = M2;
    ml = 0;
    rows = M+1;
    cols = f[0].length;
    Mat = new float[rows][cols];
    for(int i=0;i<rows;i++) 
      for(int j=0;j<cols;j++) 
	Mat[i][j] = f[i][j];
  }
  public FPoBand(FPoFull F, int M2) {
    mu = M2;
    M = M2;
    ml = 0;
    cols = F.numofCols();
    rows = M+1;
    Mat = band(F.Mat);
  }
  public FPoBand(FPoBand F) {
    super(F);
    M = F.M;
  }

  private float[][] band(float[][] f) {
    int n = f[0].length;
    float[][] ret = new float[M+1][n];
    for(int j=0;j<n;j++) {
      int i1 = Math.max(0,j-M);
      for(int i=i1;i<=j;i++) {
	int k = i-j+M;
	ret[k][j] = f[i][j];
      }
    }
    return ret;
  }
  public FPoFull poUnband() {
    FPoFull Ret = new FPoFull(cols,cols);
    for(int j=0;j<cols;j++) {
      int i1 = Math.max(0,j-mu);
      for(int i=i1;i<=j;i++) {
	int k = i-j+mu;
	Ret.Mat[i][j] = this.Mat[k][j];
      }
    }
    return Ret;
  }


  public float oneNorm() {
    
    float[] Z = new float[cols];

    for(int j=0;j<cols;j++) {
      int l = Math.min(j,M);
      int mu = Math.max(M-j,0);
      Z[j] = this.asum(l+1, 1,mu,j);
      int k = j - l;
      for(int i=mu;i<M;i++,k++) 
	Z[k] += Math.abs(this.Mat[i][j]);
    }
    float anorm = 0;//oneNorm;
    for(int j=0;j<cols;j++) 
      anorm = Math.max(anorm, Z[j]);

    return anorm;
  }

  public void factor() 
       throws SingularMatrixException
  {
    float S;

    for(int j=0;j<cols;j++) {
      S = 0;
      int ik = M;
      int jk = Math.max(j-M,0);
      int mu = Math.max(M-j,0);
      
      for(int k=mu;k<M;k++,ik--,jk++) {
	float T = this.Mat[k][j] - this.dot(k-mu,1,ik,jk,this,1,mu,j);
	T = T/this.Mat[M][jk];
	this.Mat[k][j] = T;
	S += T*T;
      }
      S = this.Mat[M][j] - S;    
      if(S <= 0) 
	throw new SingularMatrixException(j+1);
      else 
	this.Mat[M][j] = (float)Math.sqrt(S);
    }
  }

  public LNumber condition() 
       throws SingularMatrixException, WrongDataTypeException
  {
    Vector Z = new FVector(cols);
    return this.condition(Z);
  }
  public LNumber condition(Vector Ze) 
       throws SingularMatrixException, WrongDataTypeException
  {

    float[] Z = Ze.getFloatArray();
    float anorm = this.oneNorm();
   
    this.factor();
    
    float ek = 1;
    for(int j=0;j<cols;j++) 
      Z[j] = 0;
      
    for(int k=0;k<cols;k++) {
      if(Z[k] != 0) 
	ek = FUtil.signOf(ek,-Z[k]);
      if(Math.abs(ek - Z[k]) > this.Mat[M][k]) {
	float S = this.Mat[M][k]/Math.abs(ek-Z[k]);
	FUtil.scal(cols,S,Z,1);
	ek *= S;
      }
      float wk = ek - Z[k];
      float wkm = -ek - Z[k];
      float S = Math.abs(wk);
      float SM = Math.abs(wkm);
      wk = wk/this.Mat[M][k];
      wkm = wkm/this.Mat[M][k];
      int kp = k+1;
      int J2 = Math.min(kp+M,cols);
      int i = M-1;
      
      if(kp < J2) {
	for(int j=kp;j<J2;j++,i--) {
	  SM += Math.abs(Z[j] + wkm*this.Mat[i][j]);
	  Z[j] += wk*this.Mat[i][j];
	  S += Math.abs(Z[j]);
	}
	if(S < SM) {
	  float T = wkm - wk;
	  wk = wkm;
	  i = M - 1;
	  for(int j=kp;j<J2;j++,i--) 
	    Z[j] += T*this.Mat[i][j];
	}
      }
      Z[k] = wk;
    }
    float S = 1/FUtil.asum(cols,Z,1);
    FUtil.scal(cols,S,Z,1);
    
    this.solveUZ(M,Z);
    
    float ynorm = this.solveTrans(M,Z);
      
    ynorm = this.solveUZ(M,Z,ynorm);
  
    LFloat R;

    if(anorm != 0) 
      R = new LFloat(ynorm/anorm);
    else
      R = new LFloat(anorm);

    return R;
  }
  void solveUZ(int M, float[] Z) {
    for(int k=cols-1;k>=0;k--) {
      if(Math.abs(Z[k]) > Math.abs(this.Mat[M][k])) {
	float S = Math.abs(this.Mat[M][k])/Math.abs(Z[k]);
	FUtil.scal(cols,S,Z,1);
      }
      if(this.Mat[M][k] != 0)
	Z[k] /= this.Mat[M][k];
      else
	Z[k] = 1;
      int lm = Math.min(k,M);
      int la = M - lm;
      int lz = k - lm;
      float T = -Z[k];
      this.axpy(lm,T,1,la,k,Z,1,lz);
    }
    float S = (float)1.0/FUtil.asum(cols,Z,1);
    FUtil.scal(cols,S,Z,1);
  }
  private float solveTrans (int M, float[] Z) {
    
    float ynorm = 1;

    for(int k=0;k<cols;k++) {
      int LM = Math.min(k,M);
      int la = M - LM;
      int lb = k - LM;
      Z[k] -= this.dot(LM,1,la,k,Z,1,lb);
      if(Math.abs(Z[k]) > this.Mat[M][k]) {
	float S = this.Mat[M][k]/Math.abs(Z[k]);
	FUtil.scal(cols,S,Z,1);
	ynorm *= S;
      }
      Z[k] /= this.Mat[M][k];
    }
    float S = 1/FUtil.asum(cols,Z,1);
    FUtil.scal(cols,S,Z,1);
    ynorm *= S;
    
    return ynorm;
  }

  public void solve(Vector B, int J) 
       throws WrongDataTypeException
  {
    this.solve(B);
  }
  public void solve(Vector Be) 
       throws WrongDataTypeException
  {
    float[] B = Be.getFloatArray();
    this.solveTransRY(M,B);
    this.solveRX(M,B);
  }

  public Vector determ() {

    float[] Det = new float[2];
    
    Det[0] = 1;
    Det[1] = 0;
    
    for(int i=0;i<cols;i++) {
      Det[0] *= Math.pow(this.Mat[M][i],2);
      
      if(Det[0] == 0) {
	FVector V = new FVector(Det);
	return V;
      }
      else 
	FUtil.detNorm(Det);
    }
    FVector V2 = new FVector(Det);
    return V2;
  }
}
  
  
	
