/*
 ##  class NTriUp
 ##  
 ##  This class holds the routines for a upper triangular matrix stored
 ##  in full form.  There is no factor routine; triangular matrices are
 ##  already in factored form.
 ##
 */

package edu.rice.linpack.Matrix.NMatrix;

import edu.rice.linpack.util.*;
import edu.rice.linpack.LNumber.*;

public class NTriUp extends NTriFull {

  public NTriUp() {
    super();
  }
  public NTriUp(int i, int j) {
    Mat = new LNumber[i][j];
    rows = i;
    cols = j;
  }
  public NTriUp(LNumber[][] F) {
    rows = F.length;
    cols = (F[0]).length;
    Mat = F;
  }
  public NTriUp(NTriUp M) {
    rows = M.rows;
    cols = M.cols;
    Mat = new LNumber[rows][cols];
    for(int i=0;i<rows;i++)
      for(int j=0;j<cols;j++) 
	Mat[i][j] = (M.getElem(i,j)).Clone();
  }
  
  protected LNumber oneNorm() {
    LNumber tnorm = Mat[0][0].Clone();
    tnorm.setZero();
    for(int j=0;j<cols;j++) 
      tnorm.maxTo(this.asum(j+1,1,0,j));
    return tnorm;
  }
  
  public LNumber condition(LNumber[] Z) 
       throws SingularMatrixException
  {
    LNumber tnorm = this.oneNorm();
    LNumber ynorm;
    
    try {
      this.zeroDiag();
    } finally {
      
      for(int j=0;j<cols;j++) {
	Z[j] = Mat[0][0].Clone();
	Z[j].setZero();
      }
      this.solveTransUW(Z);
      
      LNumber S = (NUtil.asum(cols,Z,1)).invTo();
      NUtil.scal(cols,S,Z,1);
      
      ynorm = this.solveUZ(Z);
    }
    LNumber R;
    
    if(tnorm.equals(0))
      R = tnorm;
    else
      R = ynorm.div(tnorm);
    
    return R;
  }
  LNumber solveUZ(LNumber[] Z) {
    LNumber ynorm = Z[0].Clone();
    ynorm.setOne();
    for(int k=cols-1;k>=0;k--) {
      if((Z[k].abs()).greaterThan(Mat[k][k].abs())) {
	LNumber S = (Mat[k][k].abs()).div(Z[k].abs());
	NUtil.scal(cols,S,Z,1);
	ynorm.multTo(S);
      } 
      if(!Mat[k][k].equals(0)) 
	Z[k].divTo(Mat[k][k]);
      else 
	Z[k].setOne();

      this.axpy(k,Z[k].negate(),1,0,k,Z,1,0);
    }
    LNumber S = (NUtil.asum(cols,Z,1)).invTo();
    NUtil.scal(cols,S,Z,1);
    ynorm.multTo(S);

    return ynorm;
  }

  public void solve(LNumber[] B, int job) {

    if(job == 0) {
      int nm = cols-1;
      B[nm].divTo(Mat[nm][nm]);
      if(cols > 0) {
	for(int j=nm-1;j>=0;j--) {
	  LNumber T = B[j+1].negate();
	  this.axpy(j+1, T, 1,0,j+1,B,1,0);
	  B[j].divTo(Mat[j][j]);    
	}
      }
    }
    else {
      B[0].divTo(Mat[0][0]);
      if(cols > 1) {
	for(int j=1;j<cols;j++) {
	  B[j].subTo(this.dot(j,1,0,j,B,1,0));
	  B[j].divTo(Mat[j][j]);
	}
      }
    }
  }
  
  public void inverse() {
    for(int k=0;k<cols;k++) {
      if(Mat[k][k].equals(0))
	return;
      else {
	Mat[k][k].invTo();
	LNumber T = Mat[k][k].negate();
	this.scal(k,T,1,0,k);
	int kp = k+1;
	for(int j=kp;j<cols;j++) {
	  T = Mat[k][j].Clone();
	  Mat[k][j].setZero();
	  this.axpy(k+1,T,1,0,k,this,1,0,j);
	}
      }
    }
  }
}
